<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Users Management';

// Set current page for navigation
$current_page = 'users.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Include header
include_once ADMIN_PATH . '/includes/header.php';

// Handle search and pagination
$search = sanitizeInput($_GET['search'] ?? '');
$page = max(1, intval($_GET['page'] ?? 1));
$limit = ITEMS_PER_PAGE;
$offset = ($page - 1) * $limit;

// Build query
$where = '';
$params = [];

if (!empty($search)) {
    $where = "WHERE username LIKE ? OR email LIKE ?";
    $params = ["%$search%", "%$search%"];
}

// Get total users count
$countQuery = "SELECT COUNT(*) as total FROM users $where";
$totalUsers = fetch($countQuery, $params)['total'];
$totalPages = ceil($totalUsers / $limit);

// Get users with pagination
$query = "SELECT u.*, 
                 (SELECT COUNT(*) FROM game_plays WHERE user_id = u.id) as total_plays,
                 (SELECT COUNT(*) FROM game_plays WHERE user_id = u.id AND status = 'won') as games_won
          FROM users u 
          $where 
          ORDER BY u.created_at DESC 
          LIMIT ? OFFSET ?";

$usersParams = array_merge($params, [$limit, $offset]);
$users = fetchAll($query, $usersParams);

// Handle user deletion
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $userId = intval($_GET['delete']);
    
    // Verify CSRF token
    if (!isset($_GET['csrf_token']) || !verifyCSRFToken($_GET['csrf_token'])) {
        showNotification('Invalid request token', 'error');
    } else {
        try {
            // Start transaction
            $db = Database::getInstance();
            $db->beginTransaction();
            
            // Delete user's game plays
            delete('game_plays', 'user_id = ?', [$userId]);
            
            // Delete user
            delete('users', 'id = ?', [$userId]);
            
            // Log activity
            logActivity('delete_user', 'users', $userId, null, 'User deleted by admin');
            
            $db->commit();
            showNotification('User deleted successfully', 'success');
            
            // Redirect to refresh page
            header('Location: ' . ADMIN_URL . '/modules/users.php');
            exit;
            
        } catch (Exception $e) {
            $db->rollBack();
            showNotification('Error deleting user: ' . $e->getMessage(), 'error');
        }
    }
}
?>

<!-- Users Management Content -->
<div class="users-content">
    <!-- Display Notifications -->
    <?php displayNotifications(); ?>
    
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Users Management</h1>
        <p class="page-description">Manage all game users and their statistics</p>
    </div>
    
    <!-- Search and Filters -->
    <div class="search-section">
        <form method="GET" class="search-form">
            <div class="search-input-group">
                <input type="text" 
                       name="search" 
                       class="search-input" 
                       placeholder="Search by username or email..."
                       value="<?php echo htmlspecialchars($search); ?>">
                <button type="submit" class="btn btn-primary">
                    <span class="btn-icon">🔍</span>
                    Search
                </button>
                <?php if (!empty($search)): ?>
                <a href="<?php echo ADMIN_URL; ?>/modules/users.php" class="btn btn-secondary">
                    <span class="btn-icon">✖</span>
                    Clear
                </a>
                <?php endif; ?>
            </div>
        </form>
    </div>
    
    <!-- Users Table -->
    <div class="table-container">
        <div class="table-header">
            <h2 class="table-title">All Users (<?php echo number_format($totalUsers); ?>)</h2>
            <div class="table-actions">
                <button class="btn btn-secondary" onclick="exportUsers()">
                    <span class="btn-icon">📥</span>
                    Export
                </button>
            </div>
        </div>
        
        <?php if (empty($users)): ?>
        <div class="empty-state">
            <div class="empty-icon">👥</div>
            <h3>No users found</h3>
            <p><?php echo !empty($search) ? 'No users match your search criteria.' : 'No users have registered yet.'; ?></p>
        </div>
        <?php else: ?>
        <div class="data-table">
            <table class="users-table">
                <thead>
                    <tr>
                        <th>User</th>
                        <th>Email</th>
                        <th>Level</th>
                        <th>Coins</th>
                        <th>Games Played</th>
                        <th>Win Rate</th>
                        <th>Registered</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                    <tr>
                        <td>
                            <div class="user-info">
                                <div class="user-avatar">
                                    <span class="avatar-text"><?php echo strtoupper(substr($user['username'], 0, 2)); ?></span>
                                </div>
                                <div class="user-details">
                                    <div class="user-name"><?php echo htmlspecialchars($user['username']); ?></div>
                                    <div class="user-id">ID: #<?php echo $user['id']; ?></div>
                                </div>
                            </div>
                        </td>
                        <td>
                            <span class="email-text"><?php echo htmlspecialchars($user['email']); ?></span>
                        </td>
                        <td>
                            <span class="level-badge">Level <?php echo $user['current_level']; ?></span>
                        </td>
                        <td>
                            <div class="coins-display">
                                <span class="coins-icon">💰</span>
                                <span class="coins-value"><?php echo number_format($user['coins']); ?></span>
                            </div>
                        </td>
                        <td>
                            <div class="games-stats">
                                <span class="games-played"><?php echo number_format($user['total_plays']); ?></span>
                                <?php if ($user['games_won'] > 0): ?>
                                <span class="games-won">(<?php echo $user['games_won']; ?> won)</span>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td>
                            <?php
                            $winRate = $user['total_plays'] > 0 ? round(($user['games_won'] / $user['total_plays']) * 100, 1) : 0;
                            $rateClass = $winRate >= 70 ? 'high' : ($winRate >= 40 ? 'medium' : 'low');
                            ?>
                            <div class="win-rate">
                                <div class="rate-bar">
                                    <div class="rate-fill rate-<?php echo $rateClass; ?>" style="width: <?php echo $winRate; ?>%"></div>
                                </div>
                                <span class="rate-text"><?php echo $winRate; ?>%</span>
                            </div>
                        </td>
                        <td>
                            <span class="date-text"><?php echo formatDate($user['created_at'], 'M j, Y'); ?></span>
                        </td>
                        <td>
                            <div class="action-buttons">
                                <a href="<?php echo ADMIN_URL; ?>/modules/view-user.php?id=<?php echo $user['id']; ?>" 
                                   class="btn btn-sm btn-info" title="View Details">
                                    <span class="btn-icon">👁️</span>
                                </a>
                                <a href="<?php echo ADMIN_URL; ?>/modules/edit-user.php?id=<?php echo $user['id']; ?>" 
                                   class="btn btn-sm btn-warning" title="Edit User">
                                    <span class="btn-icon">✏️</span>
                                </a>
                                <button class="btn btn-sm btn-danger" 
                                        onclick="confirmDelete(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['username']); ?>')"
                                        title="Delete User">
                                    <span class="btn-icon">🗑️</span>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php
            $currentUrl = ADMIN_URL . '/modules/users.php';
            if (!empty($search)) {
                $currentUrl .= '?search=' . urlencode($search);
            }
            ?>
            
            <?php if ($page > 1): ?>
            <a href="<?php echo $currentUrl . (strpos($currentUrl, '?') !== false ? '&' : '?') . 'page=' . ($page - 1); ?>" 
               class="btn btn-pagination">« Previous</a>
            <?php endif; ?>
            
            <?php
            $startPage = max(1, $page - 2);
            $endPage = min($totalPages, $page + 2);
            
            for ($i = $startPage; $i <= $endPage; $i++):
            ?>
            <a href="<?php echo $currentUrl . (strpos($currentUrl, '?') !== false ? '&' : '?') . 'page=' . $i; ?>" 
               class="btn btn-pagination <?php echo $i == $page ? 'active' : ''; ?>">
                <?php echo $i; ?>
            </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="<?php echo $currentUrl . (strpos($currentUrl, '?') !== false ? '&' : '?') . 'page=' . ($page + 1); ?>" 
               class="btn btn-pagination">Next »</a>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div id="deleteModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Confirm Delete User</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <div class="modal-body">
            <p>Are you sure you want to delete user <strong id="deleteUserName"></strong>?</p>
            <p class="warning-text">This action cannot be undone and will delete all user data including game history.</p>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            <a id="deleteConfirmBtn" href="#" class="btn btn-danger">Delete User</a>
        </div>
    </div>
</div>

<script>
function confirmDelete(userId, username) {
    document.getElementById('deleteUserName').textContent = username;
    document.getElementById('deleteConfirmBtn').href = 
        '<?php echo ADMIN_URL; ?>/modules/users.php?delete=' + userId + '&csrf_token=<?php echo generateCSRFToken(); ?>';
    document.getElementById('deleteModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('deleteModal').style.display = 'none';
}

function exportUsers() {
    window.location.href = '<?php echo ADMIN_URL; ?>/modules/export-users.php';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('deleteModal');
    if (event.target == modal) {
        closeModal();
    }
}
</script>

<?php
// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
