<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Settings';

// Set current page for navigation
$current_page = 'settings.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Include header
include_once ADMIN_PATH . '/includes/header.php';

// Initialize variables
$errors = [];
$success = '';

// Get current settings
$settings = [];
$settingRows = fetchAll("SELECT setting_key, setting_value, setting_type FROM settings");
foreach ($settingRows as $row) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $errors[] = 'Invalid request token';
    } else {
        try {
            $db = Database::getInstance();
            $db->beginTransaction();
            
            // Update game settings
            $gameSettings = [
                'game_name' => sanitizeInput($_POST['game_name'] ?? 'Water Sort Game'),
                'game_version' => sanitizeInput($_POST['game_version'] ?? '1.0.0'),
                'max_levels' => intval($_POST['max_levels'] ?? 100),
                'default_coins' => intval($_POST['default_coins'] ?? 100),
                'default_hints' => intval($_POST['default_hints'] ?? 5),
                'default_undos' => intval($_POST['default_undos'] ?? 10),
                'reward_multiplier' => floatval($_POST['reward_multiplier'] ?? 1.0),
                'hints_cost' => intval($_POST['hints_cost'] ?? 50),
                'undos_cost' => intval($_POST['undos_cost'] ?? 30),
                'skip_level_cost' => intval($_POST['skip_level_cost'] ?? 100)
            ];
            
            foreach ($gameSettings as $key => $value) {
                update('settings', ['setting_value' => $value], 'setting_key = ?', [$key]);
            }
            
            // Update ads settings
            $adsSettings = [
                'ads_enabled' => isset($_POST['ads_enabled']) ? 1 : 0,
                'banner_ads_enabled' => isset($_POST['banner_ads_enabled']) ? 1 : 0,
                'interstitial_ads_enabled' => isset($_POST['interstitial_ads_enabled']) ? 1 : 0,
                'rewarded_ads_enabled' => isset($_POST['rewarded_ads_enabled']) ? 1 : 0,
                'ads_frequency' => intval($_POST['ads_frequency'] ?? 3),
                'banner_ad_unit_id' => sanitizeInput($_POST['banner_ad_unit_id'] ?? ''),
                'interstitial_ad_unit_id' => sanitizeInput($_POST['interstitial_ad_unit_id'] ?? ''),
                'rewarded_ad_unit_id' => sanitizeInput($_POST['rewarded_ad_unit_id'] ?? '')
            ];
            
            foreach ($adsSettings as $key => $value) {
                update('settings', ['setting_value' => $value], 'setting_key = ?', [$key]);
            }
            
            // Update IAP settings
            $iapSettings = [
                'iap_enabled' => isset($_POST['iap_enabled']) ? 1 : 0,
                'coins_pack_1_price' => floatval($_POST['coins_pack_1_price'] ?? 0.99),
                'coins_pack_1_coins' => intval($_POST['coins_pack_1_coins'] ?? 500),
                'coins_pack_2_price' => floatval($_POST['coins_pack_2_price'] ?? 4.99),
                'coins_pack_2_coins' => intval($_POST['coins_pack_2_coins'] ?? 3000),
                'coins_pack_3_price' => floatval($_POST['coins_pack_3_price'] ?? 9.99),
                'coins_pack_3_coins' => intval($_POST['coins_pack_3_coins'] ?? 10000),
                'hints_pack_price' => floatval($_POST['hints_pack_price'] ?? 2.99),
                'hints_pack_hints' => intval($_POST['hints_pack_hints'] ?? 25),
                'undos_pack_price' => floatval($_POST['undos_pack_price'] ?? 1.99),
                'undos_pack_undos' => intval($_POST['undos_pack_undos'] ?? 50)
            ];
            
            foreach ($iapSettings as $key => $value) {
                update('settings', ['setting_value' => $value], 'setting_key = ?', [$key]);
            }
            
            // Update notification settings
            $notificationSettings = [
                'notifications_enabled' => isset($_POST['notifications_enabled']) ? 1 : 0,
                'daily_reminder_enabled' => isset($_POST['daily_reminder_enabled']) ? 1 : 0,
                'achievement_notifications' => isset($_POST['achievement_notifications']) ? 1 : 0,
                'new_level_notifications' => isset($_POST['new_level_notifications']) ? 1 : 0,
                'fcm_server_key' => sanitizeInput($_POST['fcm_server_key'] ?? '')
            ];
            
            foreach ($notificationSettings as $key => $value) {
                update('settings', ['setting_value' => $value], 'setting_key = ?', [$key]);
            }
            
            // Update security settings
            $securitySettings = [
                'session_timeout' => intval($_POST['session_timeout'] ?? 3600),
                'max_login_attempts' => intval($_POST['max_login_attempts'] ?? 5),
                'lockout_duration' => intval($_POST['lockout_duration'] ?? 900),
                'require_email_verification' => isset($_POST['require_email_verification']) ? 1 : 0,
                'enable_two_factor' => isset($_POST['enable_two_factor']) ? 1 : 0
            ];
            
            foreach ($securitySettings as $key => $value) {
                update('settings', ['setting_value' => $value], 'setting_key = ?', [$key]);
            }
            
            // Log activity
            logActivity('update_settings', 'settings', null, null, 'Settings updated by admin');
            
            $db->commit();
            
            $success = 'Settings updated successfully!';
            
            // Refresh settings
            $settingRows = fetchAll("SELECT setting_key, setting_value FROM settings");
            $settings = [];
            foreach ($settingRows as $row) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
            
        } catch (Exception $e) {
            $db->rollBack();
            $errors[] = 'Error updating settings: ' . $e->getMessage();
        }
    }
}
?>

<!-- Settings Content -->
<div class="settings-content">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Settings</h1>
        <p class="page-description">Configure game settings and preferences</p>
    </div>
    
    <!-- Success Message -->
    <?php if (!empty($success)): ?>
    <div class="alert alert-success">
        <span class="alert-icon">✅</span>
        <span class="alert-text"><?php echo $success; ?></span>
    </div>
    <?php endif; ?>
    
    <!-- Settings Form -->
    <div class="settings-container">
        <form method="POST" class="settings-form" data-validate>
            <!-- CSRF Token -->
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <!-- Game Settings -->
            <div class="settings-section">
                <h3 class="section-title">🎮 Game Settings</h3>
                
                <div class="settings-grid">
                    <div class="form-group">
                        <label for="game_name" class="form-label">Game Name</label>
                        <input type="text" 
                               id="game_name" 
                               name="game_name" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['game_name'] ?? 'Water Sort Game'); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="game_version" class="form-label">Game Version</label>
                        <input type="text" 
                               id="game_version" 
                               name="game_version" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['game_version'] ?? '1.0.0'); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="max_levels" class="form-label">Maximum Levels</label>
                        <input type="number" 
                               id="max_levels" 
                               name="max_levels" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['max_levels'] ?? 100); ?>"
                               min="1"
                               max="9999">
                    </div>
                    
                    <div class="form-group">
                        <label for="reward_multiplier" class="form-label">Reward Multiplier</label>
                        <input type="number" 
                               id="reward_multiplier" 
                               name="reward_multiplier" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['reward_multiplier'] ?? 1.0); ?>"
                               min="0.1"
                               max="5.0"
                               step="0.1">
                    </div>
                </div>
                
                <h4 class="subsection-title">Default User Resources</h4>
                <div class="settings-grid">
                    <div class="form-group">
                        <label for="default_coins" class="form-label">Default Coins</label>
                        <input type="number" 
                               id="default_coins" 
                               name="default_coins" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['default_coins'] ?? 100); ?>"
                               min="0"
                               max="999999">
                    </div>
                    
                    <div class="form-group">
                        <label for="default_hints" class="form-label">Default Hints</label>
                        <input type="number" 
                               id="default_hints" 
                               name="default_hints" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['default_hints'] ?? 5); ?>"
                               min="0"
                               max="999">
                    </div>
                    
                    <div class="form-group">
                        <label for="default_undos" class="form-label">Default Undos</label>
                        <input type="number" 
                               id="default_undos" 
                               name="default_undos" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['default_undos'] ?? 10); ?>"
                               min="0"
                               max="999">
                    </div>
                </div>
                
                <h4 class="subsection-title">In-Game Costs</h4>
                <div class="settings-grid">
                    <div class="form-group">
                        <label for="hints_cost" class="form-label">Hints Cost (coins)</label>
                        <input type="number" 
                               id="hints_cost" 
                               name="hints_cost" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['hints_cost'] ?? 50); ?>"
                               min="0"
                               max="9999">
                    </div>
                    
                    <div class="form-group">
                        <label for="undos_cost" class="form-label">Undos Cost (coins)</label>
                        <input type="number" 
                               id="undos_cost" 
                               name="undos_cost" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['undos_cost'] ?? 30); ?>"
                               min="0"
                               max="9999">
                    </div>
                    
                    <div class="form-group">
                        <label for="skip_level_cost" class="form-label">Skip Level Cost (coins)</label>
                        <input type="number" 
                               id="skip_level_cost" 
                               name="skip_level_cost" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['skip_level_cost'] ?? 100); ?>"
                               min="0"
                               max="9999">
                    </div>
                </div>
            </div>
            
            <!-- Ads Settings -->
            <div class="settings-section">
                <h3 class="section-title">📱 Advertisement Settings</h3>
                
                <div class="toggle-group">
                    <label class="toggle-label">
                        <input type="checkbox" name="ads_enabled" class="toggle-input" 
                               <?php echo ($settings['ads_enabled'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                        <span class="toggle-text">Enable Ads</span>
                    </label>
                </div>
                
                <div class="ads-types">
                    <label class="toggle-label">
                        <input type="checkbox" name="banner_ads_enabled" class="toggle-input" 
                               <?php echo ($settings['banner_ads_enabled'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                        <span class="toggle-text">Banner Ads</span>
                    </label>
                    
                    <label class="toggle-label">
                        <input type="checkbox" name="interstitial_ads_enabled" class="toggle-input" 
                               <?php echo ($settings['interstitial_ads_enabled'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                        <span class="toggle-text">Interstitial Ads</span>
                    </label>
                    
                    <label class="toggle-label">
                        <input type="checkbox" name="rewarded_ads_enabled" class="toggle-input" 
                               <?php echo ($settings['rewarded_ads_enabled'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                        <span class="toggle-text">Rewarded Ads</span>
                    </label>
                </div>
                
                <div class="settings-grid">
                    <div class="form-group">
                        <label for="ads_frequency" class="form-label">Ads Frequency (every N plays)</label>
                        <input type="number" 
                               id="ads_frequency" 
                               name="ads_frequency" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['ads_frequency'] ?? 3); ?>"
                               min="1"
                               max="20">
                    </div>
                    
                    <div class="form-group full-width">
                        <label for="banner_ad_unit_id" class="form-label">Banner Ad Unit ID</label>
                        <input type="text" 
                               id="banner_ad_unit_id" 
                               name="banner_ad_unit_id" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['banner_ad_unit_id'] ?? ''); ?>"
                               placeholder="ca-app-pub-XXXXXXXXXXXXXXXXX/XXXXXXXXXX">
                    </div>
                    
                    <div class="form-group full-width">
                        <label for="interstitial_ad_unit_id" class="form-label">Interstitial Ad Unit ID</label>
                        <input type="text" 
                               id="interstitial_ad_unit_id" 
                               name="interstitial_ad_unit_id" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['interstitial_ad_unit_id'] ?? ''); ?>"
                               placeholder="ca-app-pub-XXXXXXXXXXXXXXXXX/XXXXXXXXXX">
                    </div>
                    
                    <div class="form-group full-width">
                        <label for="rewarded_ad_unit_id" class="form-label">Rewarded Ad Unit ID</label>
                        <input type="text" 
                               id="rewarded_ad_unit_id" 
                               name="rewarded_ad_unit_id" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['rewarded_ad_unit_id'] ?? ''); ?>"
                               placeholder="ca-app-pub-XXXXXXXXXXXXXXXXX/XXXXXXXXXX">
                    </div>
                </div>
            </div>
            
            <!-- IAP Settings -->
            <div class="settings-section">
                <h3 class="section-title">💰 In-App Purchase Settings</h3>
                
                <div class="toggle-group">
                    <label class="toggle-label">
                        <input type="checkbox" name="iap_enabled" class="toggle-input" 
                               <?php echo ($settings['iap_enabled'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                        <span class="toggle-text">Enable In-App Purchases</span>
                    </label>
                </div>
                
                <h4 class="subsection-title">Coins Packs</h4>
                <div class="iap-grid">
                    <div class="iap-item">
                        <h5>Pack 1</h5>
                        <div class="form-group">
                            <label class="form-label">Price ($)</label>
                            <input type="number" 
                                   name="coins_pack_1_price" 
                                   class="form-input" 
                                   value="<?php echo htmlspecialchars($settings['coins_pack_1_price'] ?? 0.99); ?>"
                                   min="0.99"
                                   max="99.99"
                                   step="0.01">
                        </div>
                        <div class="form-group">
                            <label class="form-label">Coins</label>
                            <input type="number" 
                                   name="coins_pack_1_coins" 
                                   class="form-input" 
                                   value="<?php echo htmlspecialchars($settings['coins_pack_1_coins'] ?? 500); ?>"
                                   min="1"
                                   max="999999">
                        </div>
                    </div>
                    
                    <div class="iap-item">
                        <h5>Pack 2</h5>
                        <div class="form-group">
                            <label class="form-label">Price ($)</label>
                            <input type="number" 
                                   name="coins_pack_2_price" 
                                   class="form-input" 
                                   value="<?php echo htmlspecialchars($settings['coins_pack_2_price'] ?? 4.99); ?>"
                                   min="0.99"
                                   max="99.99"
                                   step="0.01">
                        </div>
                        <div class="form-group">
                            <label class="form-label">Coins</label>
                            <input type="number" 
                                   name="coins_pack_2_coins" 
                                   class="form-input" 
                                   value="<?php echo htmlspecialchars($settings['coins_pack_2_coins'] ?? 3000); ?>"
                                   min="1"
                                   max="999999">
                        </div>
                    </div>
                    
                    <div class="iap-item">
                        <h5>Pack 3</h5>
                        <div class="form-group">
                            <label class="form-label">Price ($)</label>
                            <input type="number" 
                                   name="coins_pack_3_price" 
                                   class="form-input" 
                                   value="<?php echo htmlspecialchars($settings['coins_pack_3_price'] ?? 9.99); ?>"
                                   min="0.99"
                                   max="99.99"
                                   step="0.01">
                        </div>
                        <div class="form-group">
                            <label class="form-label">Coins</label>
                            <input type="number" 
                                   name="coins_pack_3_coins" 
                                   class="form-input" 
                                   value="<?php echo htmlspecialchars($settings['coins_pack_3_coins'] ?? 10000); ?>"
                                   min="1"
                                   max="999999">
                        </div>
                    </div>
                </div>
                
                <h4 class="subsection-title">Power-ups</h4>
                <div class="settings-grid">
                    <div class="form-group">
                        <label class="form-label">Hints Pack Price ($)</label>
                        <input type="number" 
                               name="hints_pack_price" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['hints_pack_price'] ?? 2.99); ?>"
                               min="0.99"
                               max="99.99"
                               step="0.01">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Hints Pack Quantity</label>
                        <input type="number" 
                               name="hints_pack_hints" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['hints_pack_hints'] ?? 25); ?>"
                               min="1"
                               max="999">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Undos Pack Price ($)</label>
                        <input type="number" 
                               name="undos_pack_price" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['undos_pack_price'] ?? 1.99); ?>"
                               min="0.99"
                               max="99.99"
                               step="0.01">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Undos Pack Quantity</label>
                        <input type="number" 
                               name="undos_pack_undos" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['undos_pack_undos'] ?? 50); ?>"
                               min="1"
                               max="999">
                    </div>
                </div>
            </div>
            
            <!-- Security Settings -->
            <div class="settings-section">
                <h3 class="section-title">🔐 Security Settings</h3>
                
                <div class="settings-grid">
                    <div class="form-group">
                        <label for="session_timeout" class="form-label">Session Timeout (seconds)</label>
                        <input type="number" 
                               id="session_timeout" 
                               name="session_timeout" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['session_timeout'] ?? 3600); ?>"
                               min="300"
                               max="86400">
                    </div>
                    
                    <div class="form-group">
                        <label for="max_login_attempts" class="form-label">Max Login Attempts</label>
                        <input type="number" 
                               id="max_login_attempts" 
                               name="max_login_attempts" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['max_login_attempts'] ?? 5); ?>"
                               min="3"
                               max="10">
                    </div>
                    
                    <div class="form-group">
                        <label for="lockout_duration" class="form-label">Lockout Duration (seconds)</label>
                        <input type="number" 
                               id="lockout_duration" 
                               name="lockout_duration" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($settings['lockout_duration'] ?? 900); ?>"
                               min="60"
                               max="3600">
                    </div>
                </div>
                
                <div class="security-options">
                    <label class="toggle-label">
                        <input type="checkbox" name="require_email_verification" class="toggle-input" 
                               <?php echo ($settings['require_email_verification'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                        <span class="toggle-text">Require Email Verification</span>
                    </label>
                    
                    <label class="toggle-label">
                        <input type="checkbox" name="enable_two_factor" class="toggle-input" 
                               <?php echo ($settings['enable_two_factor'] ?? 0) ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                        <span class="toggle-text">Enable Two-Factor Authentication</span>
                    </label>
                </div>
            </div>
            
            <!-- Form Actions -->
            <div class="form-actions">
                <button type="submit" class="btn btn-primary">
                    <span class="btn-icon">💾</span>
                    Save Settings
                </button>
                <button type="button" class="btn btn-secondary" onclick="resetSettings()">
                    <span class="btn-icon">🔄</span>
                    Reset to Defaults
                </button>
                <button type="button" class="btn btn-secondary" onclick="exportSettings()">
                    <span class="btn-icon">📥</span>
                    Export Settings
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function resetSettings() {
    if (!confirm('Are you sure you want to reset all settings to default values?')) return;
    
    fetch('<?php echo ADMIN_URL; ?>/modules/reset-settings.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({
            csrf_token: '<?php echo generateCSRFToken(); ?>'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Settings reset to defaults successfully', 'success');
            location.reload();
        } else {
            showNotification(data.message || 'Error resetting settings', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Error resetting settings', 'error');
    });
}

function exportSettings() {
    fetch('<?php echo ADMIN_URL; ?>/modules/export-settings.php')
        .then(response => response.blob())
        .then(blob => {
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'game-settings-' + new Date().toISOString().slice(0, 10) + '.json';
            document.body.appendChild(a);
            a.click();
            window.URL.revokeObjectURL(url);
            document.body.removeChild(a);
            showNotification('Settings exported successfully', 'success');
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Error exporting settings', 'error');
        });
}
</script>

<style>
.settings-container {
    max-width: 1000px;
    margin: 0 auto;
}

.settings-section {
    background: white;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    margin-bottom: 30px;
    overflow: hidden;
}

.section-title {
    padding: 20px;
    background: #f8f9fa;
    margin: 0;
    border-bottom: 1px solid #e9ecef;
    font-size: 18px;
    font-weight: 600;
    color: #333;
}

.subsection-title {
    padding: 20px 20px 10px;
    margin: 0;
    font-size: 16px;
    font-weight: 500;
    color: #666;
}

.settings-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    padding: 20px;
}

.form-group.full-width {
    grid-column: 1 / -1;
}

.toggle-group {
    padding: 20px;
    border-bottom: 1px solid #f0f0f0;
}

.toggle-label {
    display: flex;
    align-items: center;
    gap: 10px;
    cursor: pointer;
    margin-bottom: 10px;
}

.toggle-input {
    display: none;
}

.toggle-slider {
    width: 50px;
    height: 24px;
    background: #ccc;
    border-radius: 12px;
    position: relative;
    transition: background 0.3s;
}

.toggle-slider::before {
    content: '';
    position: absolute;
    width: 20px;
    height: 20px;
    background: white;
    border-radius: 50%;
    top: 2px;
    left: 2px;
    transition: transform 0.3s;
}

.toggle-input:checked + .toggle-slider {
    background: #667eea;
}

.toggle-input:checked + .toggle-slider::before {
    transform: translateX(26px);
}

.toggle-text {
    font-weight: 500;
    color: #333;
}

.ads-types {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    padding: 0 20px 20px;
}

.iap-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    padding: 0 20px 20px;
}

.iap-item {
    border: 1px solid #e9ecef;
    border-radius: 8px;
    padding: 15px;
}

.iap-item h5 {
    margin: 0 0 15px 0;
    font-weight: 600;
    color: #333;
}

.security-options {
    padding: 0 20px 20px;
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.form-actions {
    padding: 20px;
    background: #f8f9fa;
    display: flex;
    gap: 15px;
    justify-content: flex-start;
}
</style>

<?php
// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
