<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Reports';

// Set current page for navigation
$current_page = 'reports.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Include header
include_once ADMIN_PATH . '/includes/header.php';

// Handle filters
$dateFilter = sanitizeInput($_GET['date_filter'] ?? '7'); // 7, 30, 90 days
$levelFilter = intval($_GET['level_filter'] ?? 0);
$userFilter = sanitizeInput($_GET['user_filter'] ?? '');
$statusFilter = sanitizeInput($_GET['status_filter'] ?? '');

// Build WHERE clause
$where = [];
$params = [];

// Date filter
if ($dateFilter === '7') {
    $where[] = "DATE(gp.created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)";
} elseif ($dateFilter === '30') {
    $where[] = "DATE(gp.created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
} elseif ($dateFilter === '90') {
    $where[] = "DATE(gp.created_at) >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";
}

// Level filter
if ($levelFilter > 0) {
    $where[] = "gp.level_id = ?";
    $params[] = $levelFilter;
}

// User filter
if (!empty($userFilter)) {
    $where[] = "(u.username LIKE ? OR u.email LIKE ?)";
    $params[] = "%$userFilter%";
    $params[] = "%$userFilter%";
}

// Status filter
if (!empty($statusFilter)) {
    $where[] = "gp.status = ?";
    $params[] = $statusFilter;
}

$whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$limit = ITEMS_PER_PAGE;
$offset = ($page - 1) * $limit;

// Get total records
$countQuery = "SELECT COUNT(*) as total 
               FROM game_plays gp 
               LEFT JOIN users u ON gp.user_id = u.id 
               $whereClause";
$totalRecords = fetch($countQuery, $params)['total'];
$totalPages = ceil($totalRecords / $limit);

// Get game plays with details
$query = "SELECT gp.*, 
                u.username, 
                u.email,
                l.level_number,
                l.name as level_name,
                l.difficulty,
                TIMESTAMPDIFF(SECOND, gp.started_at, COALESCE(gp.completed_at, NOW())) as duration_seconds
         FROM game_plays gp
         LEFT JOIN users u ON gp.user_id = u.id
         LEFT JOIN levels l ON gp.level_id = l.id
         $whereClause
         ORDER BY gp.started_at DESC
         LIMIT ? OFFSET ?";

$playsParams = array_merge($params, [$limit, $offset]);
$gamePlays = fetchAll($query, $playsParams);

// Get filter options
$levels = fetchAll("SELECT id, level_number, name FROM levels ORDER BY level_number");
$statusOptions = ['playing', 'won', 'lost', 'quit'];

// Get summary statistics
$summaryQuery = "SELECT 
                    COUNT(*) as total_plays,
                    SUM(CASE WHEN gp.status = 'won' THEN 1 ELSE 0 END) as wins,
                    SUM(CASE WHEN gp.status = 'lost' THEN 1 ELSE 0 END) as losses,
                    SUM(CASE WHEN gp.status = 'quit' THEN 1 ELSE 0 END) as quits,
                    AVG(gp.moves_count) as avg_moves,
                    AVG(gp.coins_earned) as avg_coins,
                    AVG(TIMESTAMPDIFF(SECOND, gp.started_at, COALESCE(gp.completed_at, NOW()))) as avg_duration
                FROM game_plays gp 
                LEFT JOIN users u ON gp.user_id = u.id 
                $whereClause";

$summary = fetch($summaryQuery, $params);
?>

<!-- Reports Content -->
<div class="reports-content">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Game Reports</h1>
        <p class="page-description">Detailed reports of game plays and user activity</p>
    </div>
    
    <!-- Summary Cards -->
    <div class="summary-cards">
        <div class="summary-card">
            <div class="summary-icon">
                <span class="icon">🎮</span>
            </div>
            <div class="summary-content">
                <h3 class="summary-title">Total Plays</h3>
                <p class="summary-value"><?php echo number_format($summary['total_plays'] ?? 0); ?></p>
            </div>
        </div>
        
        <div class="summary-card">
            <div class="summary-icon">
                <span class="icon">🏆</span>
            </div>
            <div class="summary-content">
                <h3 class="summary-title">Wins</h3>
                <p class="summary-value"><?php echo number_format($summary['wins'] ?? 0); ?></p>
                <div class="summary-detail">
                    Win Rate: <?php echo $summary['total_plays'] > 0 ? round(($summary['wins'] / $summary['total_plays']) * 100, 1) : 0; ?>%
                </div>
            </div>
        </div>
        
        <div class="summary-card">
            <div class="summary-icon">
                <span class="icon">🎯</span>
            </div>
            <div class="summary-content">
                <h3 class="summary-title">Avg Moves</h3>
                <p class="summary-value"><?php echo round($summary['avg_moves'] ?? 0, 1); ?></p>
            </div>
        </div>
        
        <div class="summary-card">
            <div class="summary-icon">
                <span class="icon">⏱️</span>
            </div>
            <div class="summary-content">
                <h3 class="summary-title">Avg Duration</h3>
                <p class="summary-value"><?php echo formatDuration($summary['avg_duration'] ?? 0); ?></p>
            </div>
        </div>
    </div>
    
    <!-- Filters -->
    <div class="filters-section">
        <form method="GET" class="filters-form">
            <div class="filter-row">
                <div class="filter-group">
                    <label for="date_filter" class="filter-label">Date Range</label>
                    <select id="date_filter" name="date_filter" class="filter-select">
                        <option value="7" <?php echo $dateFilter === '7' ? 'selected' : ''; ?>>Last 7 Days</option>
                        <option value="30" <?php echo $dateFilter === '30' ? 'selected' : ''; ?>>Last 30 Days</option>
                        <option value="90" <?php echo $dateFilter === '90' ? 'selected' : ''; ?>>Last 90 Days</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label for="level_filter" class="filter-label">Level</label>
                    <select id="level_filter" name="level_filter" class="filter-select">
                        <option value="0">All Levels</option>
                        <?php foreach ($levels as $level): ?>
                        <option value="<?php echo $level['id']; ?>" <?php echo $levelFilter == $level['id'] ? 'selected' : ''; ?>>
                            Level <?php echo $level['level_number']; ?> - <?php echo htmlspecialchars($level['name']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label for="user_filter" class="filter-label">User</label>
                    <input type="text" 
                           id="user_filter" 
                           name="user_filter" 
                           class="filter-input" 
                           placeholder="Username or email"
                           value="<?php echo htmlspecialchars($userFilter); ?>">
                </div>
                
                <div class="filter-group">
                    <label for="status_filter" class="filter-label">Status</label>
                    <select id="status_filter" name="status_filter" class="filter-select">
                        <option value="">All Status</option>
                        <?php foreach ($statusOptions as $status): ?>
                        <option value="<?php echo $status; ?>" <?php echo $statusFilter === $status ? 'selected' : ''; ?>>
                            <?php echo ucfirst($status); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn btn-primary">
                        <span class="btn-icon">🔍</span>
                        Apply Filters
                    </button>
                    <a href="<?php echo ADMIN_URL; ?>/modules/reports.php" class="btn btn-secondary">
                        <span class="btn-icon">✖</span>
                        Clear
                    </a>
                    <button type="button" class="btn btn-secondary" onclick="exportReports()">
                        <span class="btn-icon">📥</span>
                        Export
                    </button>
                </div>
            </div>
        </form>
    </div>
    
    <!-- Reports Table -->
    <div class="table-container">
        <div class="table-header">
            <h2 class="table-title">Game Plays (<?php echo number_format($totalRecords); ?>)</h2>
            <div class="table-actions">
                <button class="btn btn-sm btn-secondary" onclick="refreshReports()">
                    <span class="btn-icon">🔄</span>
                    Refresh
                </button>
            </div>
        </div>
        
        <?php if (empty($gamePlays)): ?>
        <div class="empty-state">
            <div class="empty-icon">📊</div>
            <h3>No game plays found</h3>
            <p>No game plays match your current filters.</p>
        </div>
        <?php else: ?>
        <div class="data-table">
            <table class="reports-table">
                <thead>
                    <tr>
                        <th>Date/Time</th>
                        <th>User</th>
                        <th>Level</th>
                        <th>Status</th>
                        <th>Moves</th>
                        <th>Time</th>
                        <th>Coins</th>
                        <th>Usage</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($gamePlays as $play): ?>
                    <tr>
                        <td>
                            <div class="datetime-info">
                                <div class="date"><?php echo formatDate($play['started_at'], 'M j, Y'); ?></div>
                                <div class="time"><?php echo formatDate($play['started_at'], 'g:i A'); ?></div>
                            </div>
                        </td>
                        <td>
                            <div class="user-info">
                                <div class="user-name"><?php echo htmlspecialchars($play['username'] ?? 'Guest'); ?></div>
                                <div class="user-email"><?php echo htmlspecialchars($play['email'] ?? ''); ?></div>
                            </div>
                        </td>
                        <td>
                            <div class="level-info">
                                <div class="level-number">Level <?php echo $play['level_number'] ?? 'N/A'; ?></div>
                                <div class="level-name"><?php echo htmlspecialchars($play['level_name'] ?? ''); ?></div>
                                <span class="difficulty-badge difficulty-<?php echo strtolower($play['difficulty'] ?? 'easy'); ?>">
                                    <?php echo ucfirst($play['difficulty'] ?? 'Easy'); ?>
                                </span>
                            </div>
                        </td>
                        <td>
                            <span class="status-badge status-<?php echo $play['status']; ?>">
                                <?php
                                $statusIcons = [
                                    'playing' => '🎮',
                                    'won' => '🏆',
                                    'lost' => '😞',
                                    'quit' => '🚪'
                                ];
                                echo ($statusIcons[$play['status']] ?? '📝') . ' ' . ucfirst($play['status']);
                                ?>
                            </span>
                        </td>
                        <td>
                            <span class="moves-count"><?php echo number_format($play['moves_count']); ?></span>
                        </td>
                        <td>
                            <span class="duration"><?php echo formatDuration($play['duration_seconds']); ?></span>
                        </td>
                        <td>
                            <div class="coins-earned">
                                <?php if ($play['coins_earned'] > 0): ?>
                                <span class="coins-positive">+<?php echo number_format($play['coins_earned']); ?></span>
                                <?php else: ?>
                                <span class="coins-zero">0</span>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td>
                            <div class="usage-stats">
                                <div class="usage-item">
                                    <span class="usage-label">Hints:</span>
                                    <span class="usage-value"><?php echo $play['hints_used']; ?></span>
                                </div>
                                <div class="usage-item">
                                    <span class="usage-label">Undos:</span>
                                    <span class="usage-value"><?php echo $play['undos_used']; ?></span>
                                </div>
                            </div>
                        </td>
                        <td>
                            <div class="action-buttons">
                                <button class="btn btn-sm btn-info" 
                                        onclick="viewPlayDetails(<?php echo $play['id']; ?>)"
                                        title="View Details">
                                    <span class="btn-icon">👁️</span>
                                </button>
                                <?php if ($play['status'] === 'playing'): ?>
                                <button class="btn btn-sm btn-warning" 
                                        onclick="markAsCompleted(<?php echo $play['id']; ?>)"
                                        title="Mark as Completed">
                                    <span class="btn-icon">✅</span>
                                </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
        <div class="pagination">
            <?php
            $currentUrl = ADMIN_URL . '/modules/reports.php';
            $queryParams = [];
            if ($dateFilter !== '7') $queryParams['date_filter'] = $dateFilter;
            if ($levelFilter > 0) $queryParams['level_filter'] = $levelFilter;
            if (!empty($userFilter)) $queryParams['user_filter'] = $userFilter;
            if (!empty($statusFilter)) $queryParams['status_filter'] = $statusFilter;
            
            $urlString = $currentUrl . (!empty($queryParams) ? '?' . http_build_query($queryParams) : '');
            ?>
            
            <?php if ($page > 1): ?>
            <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . ($page - 1); ?>" 
               class="btn btn-pagination">« Previous</a>
            <?php endif; ?>
            
            <?php
            $startPage = max(1, $page - 2);
            $endPage = min($totalPages, $page + 2);
            
            for ($i = $startPage; $i <= $endPage; $i++):
            ?>
            <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . $i; ?>" 
               class="btn btn-pagination <?php echo $i == $page ? 'active' : ''; ?>">
                <?php echo $i; ?>
            </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="<?php echo $urlString . (strpos($urlString, '?') !== false ? '&' : '?') . 'page=' . ($page + 1); ?>" 
               class="btn btn-pagination">Next »</a>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- Play Details Modal -->
<div id="playDetailsModal" class="modal" style="display: none;">
    <div class="modal-content modal-large">
        <div class="modal-header">
            <h3>Game Play Details</h3>
            <button class="modal-close" onclick="closeDetailsModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div id="playDetailsContent">
                <!-- Details will be loaded here -->
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeDetailsModal()">Close</button>
        </div>
    </div>
</div>

<script>
function viewPlayDetails(playId) {
    fetch('<?php echo ADMIN_URL; ?>/modules/play-details.php?id=' + playId)
        .then(response => response.text())
        .then(html => {
            document.getElementById('playDetailsContent').innerHTML = html;
            document.getElementById('playDetailsModal').style.display = 'block';
        })
        .catch(error => {
            console.error('Error loading play details:', error);
            showNotification('Error loading play details', 'error');
        });
}

function closeDetailsModal() {
    document.getElementById('playDetailsModal').style.display = 'none';
}

function markAsCompleted(playId) {
    if (!confirm('Mark this game as completed?')) return;
    
    const formData = new FormData();
    formData.append('play_id', playId);
    formData.append('csrf_token', '<?php echo generateCSRFToken(); ?>');
    
    fetch('<?php echo ADMIN_URL; ?>/modules/mark-completed.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Game marked as completed', 'success');
            location.reload();
        } else {
            showNotification(data.message || 'Error marking game as completed', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Error marking game as completed', 'error');
    });
}

function refreshReports() {
    location.reload();
}

function exportReports() {
    const params = new URLSearchParams(window.location.search);
    params.set('export', '1');
    window.location.href = '<?php echo ADMIN_URL; ?>/modules/export-reports.php?' + params.toString();
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('playDetailsModal');
    if (event.target == modal) {
        closeDetailsModal();
    }
}

// Helper function to format duration
function formatDuration(seconds) {
    if (!seconds || seconds <= 0) return '0:00';
    
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = Math.floor(seconds % 60);
    
    if (hours > 0) {
        return hours + ':' + minutes.toString().padStart(2, '0') + ':' + secs.toString().padStart(2, '0');
    } else {
        return minutes + ':' + secs.toString().padStart(2, '0');
    }
}
</script>

<?php
// Helper function to format duration
function formatDuration($seconds) {
    if (!$seconds || $seconds <= 0) return '0:00';
    
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);
    $secs = floor($seconds % 60);
    
    if ($hours > 0) {
        return $hours . ':' . str_pad($minutes, 2, '0', STR_PAD_LEFT) . ':' . str_pad($secs, 2, '0', STR_PAD_LEFT);
    } else {
        return $minutes . ':' . str_pad($secs, 2, '0', STR_PAD_LEFT);
    }
}

// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
