<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Dashboard';

// Set current page for navigation
$current_page = 'dashboard.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Set page title
$page_title = 'Dashboard';

// Get dashboard statistics
$stats = getDashboardStats();
$recent_users = getRecentUsers();
$popular_levels = getPopularLevels();
$recent_activity = getRecentActivity();

/**
 * Get dashboard statistics
 */
function getDashboardStats() {
    $db = Database::getInstance();
    $stats = [];
    
    try {
        // Total users
        $stmt = $db->prepare("SELECT COUNT(*) as total FROM users");
        $stmt->execute();
        $stats['total_users'] = $stmt->fetchColumn();
        
        // Total levels
        $stmt = $db->prepare("SELECT COUNT(*) as total FROM levels");
        $stmt->execute();
        $stats['total_levels'] = $stmt->fetchColumn();
        
        // Total game plays
        $stmt = $db->prepare("SELECT COUNT(*) as total FROM game_plays");
        $stmt->execute();
        $stats['total_plays'] = $stmt->fetchColumn();
        
        // Active users today
        $stmt = $db->prepare("SELECT COUNT(DISTINCT user_id) as total FROM game_plays WHERE DATE(created_at) = CURDATE()");
        $stmt->execute();
        $stats['active_today'] = $stmt->fetchColumn();
        
        // New users this week
        $stmt = $db->prepare("SELECT COUNT(*) as total FROM users WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)");
        $stmt->execute();
        $stats['new_users_week'] = $stmt->fetchColumn();
        
        // Total revenue (if applicable)
        $stmt = $db->prepare("SELECT SUM(amount) as total FROM purchases WHERE status = 'completed'");
        $stmt->execute();
        $stats['total_revenue'] = $stmt->fetchColumn() ?: 0;
        
    } catch (Exception $e) {
        error_log("Dashboard stats error: " . $e->getMessage());
        $stats = [
            'total_users' => 0,
            'total_levels' => 0,
            'total_plays' => 0,
            'active_today' => 0,
            'new_users_week' => 0,
            'total_revenue' => 0
        ];
    }
    
    return $stats;
}

/**
 * Get recent users
 */
function getRecentUsers($limit = 5) {
    $db = Database::getInstance();
    $users = [];
    
    try {
        $stmt = $db->prepare("
            SELECT id, username, email, coins, level, created_at 
            FROM users 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Recent users error: " . $e->getMessage());
    }
    
    return $users;
}

/**
 * Get popular levels
 */
function getPopularLevels($limit = 5) {
    $db = Database::getInstance();
    $levels = [];
    
    try {
        $stmt = $db->prepare("
            SELECT l.id, l.level_number, l.title, l.difficulty, COUNT(gp.id) as play_count
            FROM levels l
            LEFT JOIN game_plays gp ON l.id = gp.level_id
            GROUP BY l.id, l.level_number, l.title, l.difficulty
            ORDER BY play_count DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        $levels = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Popular levels error: " . $e->getMessage());
    }
    
    return $levels;
}

/**
 * Get recent activity
 */
function getRecentActivity($limit = 10) {
    $db = Database::getInstance();
    $activities = [];
    
    try {
        $stmt = $db->prepare("
            SELECT al.action, al.details, al.created_at,
                   CASE 
                       WHEN al.user_id > 0 THEN u.username
                       WHEN al.admin_id > 0 THEN a.username
                       ELSE 'System'
                   END as actor_name
            FROM activity_logs al
            LEFT JOIN users u ON al.user_id = u.id
            LEFT JOIN admins a ON al.admin_id = a.id
            ORDER BY al.created_at DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        $activities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Recent activity error: " . $e->getMessage());
    }
    
    return $activities;
}

// Include header
include_once ADMIN_PATH . '/includes/header.php';
?>

<!-- Dashboard Content -->
<div class="dashboard-content">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Dashboard</h1>
        <p class="page-description">Welcome to the Water Sort Game admin panel. Here's an overview of your game's performance.</p>
    </div>
    
    <!-- Statistics Cards -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon">
                <span class="icon">👥</span>
            </div>
            <div class="stat-content">
                <h3 class="stat-title">Total Users</h3>
                <p class="stat-value"><?php echo number_format($stats['total_users']); ?></p>
                <div class="stat-change positive">
                    <span class="change-icon">📈</span>
                    <span class="change-text">+<?php echo $stats['new_users_week']; ?> this week</span>
                </div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">
                <span class="icon">🎮</span>
            </div>
            <div class="stat-content">
                <h3 class="stat-title">Total Levels</h3>
                <p class="stat-value"><?php echo number_format($stats['total_levels']); ?></p>
                <div class="stat-change neutral">
                    <span class="change-icon">📊</span>
                    <span class="change-text">Game levels</span>
                </div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">
                <span class="icon">🎯</span>
            </div>
            <div class="stat-content">
                <h3 class="stat-title">Total Plays</h3>
                <p class="stat-value"><?php echo number_format($stats['total_plays']); ?></p>
                <div class="stat-change positive">
                    <span class="change-icon">🔥</span>
                    <span class="change-text"><?php echo $stats['active_today']; ?> active today</span>
                </div>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">
                <span class="icon">💰</span>
            </div>
            <div class="stat-content">
                <h3 class="stat-title">Total Revenue</h3>
                <p class="stat-value">$<?php echo number_format($stats['total_revenue'], 2); ?></p>
                <div class="stat-change neutral">
                    <span class="change-icon">💵</span>
                    <span class="change-text">All time</span>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Dashboard Grid -->
    <div class="dashboard-grid">
        <!-- Recent Users -->
        <div class="dashboard-card">
            <div class="card-header">
                <h2 class="card-title">Recent Users</h2>
                <a href="<?php echo ADMIN_URL; ?>/modules/users.php" class="card-action">View All</a>
            </div>
            <div class="card-content">
                <?php if (empty($recent_users)): ?>
                <div class="empty-state">
                    <p>No users found.</p>
                </div>
                <?php else: ?>
                <div class="user-list">
                    <?php foreach ($recent_users as $user): ?>
                    <div class="user-item">
                        <div class="user-avatar">
                            <span class="avatar-text"><?php echo strtoupper(substr($user['username'], 0, 2)); ?></span>
                        </div>
                        <div class="user-info">
                            <h4 class="user-name"><?php echo htmlspecialchars($user['username']); ?></h4>
                            <p class="user-email"><?php echo htmlspecialchars($user['email']); ?></p>
                        </div>
                        <div class="user-stats">
                            <span class="user-level">Level <?php echo $user['level']; ?></span>
                            <span class="user-coins"><?php echo number_format($user['coins']); ?> coins</span>
                        </div>
                        <div class="user-time">
                            <span class="time-text"><?php echo formatDate($user['created_at']); ?></span>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Popular Levels -->
        <div class="dashboard-card">
            <div class="card-header">
                <h2 class="card-title">Popular Levels</h2>
                <a href="<?php echo ADMIN_URL; ?>/modules/levels.php" class="card-action">Manage</a>
            </div>
            <div class="card-content">
                <?php if (empty($popular_levels)): ?>
                <div class="empty-state">
                    <p>No levels found.</p>
                </div>
                <?php else: ?>
                <div class="level-list">
                    <?php foreach ($popular_levels as $level): ?>
                    <div class="level-item">
                        <div class="level-info">
                            <h4 class="level-title">Level <?php echo $level['level_number']; ?></h4>
                            <p class="level-name"><?php echo htmlspecialchars($level['title']); ?></p>
                        </div>
                        <div class="level-stats">
                            <span class="difficulty-badge difficulty-<?php echo strtolower($level['difficulty']); ?>">
                                <?php echo ucfirst($level['difficulty']); ?>
                            </span>
                            <span class="play-count"><?php echo number_format($level['play_count']); ?> plays</span>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Recent Activity -->
        <div class="dashboard-card full-width">
            <div class="card-header">
                <h2 class="card-title">Recent Activity</h2>
                <a href="<?php echo ADMIN_URL; ?>/modules/logs.php" class="card-action">View All</a>
            </div>
            <div class="card-content">
                <?php if (empty($recent_activity)): ?>
                <div class="empty-state">
                    <p>No recent activity found.</p>
                </div>
                <?php else: ?>
                <div class="activity-list">
                    <?php foreach ($recent_activity as $activity): ?>
                    <div class="activity-item">
                        <div class="activity-icon">
                            <?php
                            $icon = '📝';
                            switch ($activity['action']) {
                                case 'user_login': $icon = '👤'; break;
                                case 'user_register': $icon = '➕'; break;
                                case 'level_complete': $icon = '🎯'; break;
                                case 'purchase': $icon = '💰'; break;
                                case 'admin_login': $icon = '🔐'; break;
                                case 'admin_logout': $icon = '🚪'; break;
                            }
                            echo '<span class="icon">' . $icon . '</span>';
                            ?>
                        </div>
                        <div class="activity-content">
                            <p class="activity-text">
                                <strong><?php echo htmlspecialchars($activity['actor_name']); ?></strong>
                                <?php echo htmlspecialchars($activity['details']); ?>
                            </p>
                            <span class="activity-time"><?php echo formatDateTime($activity['created_at']); ?></span>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Quick Actions -->
    <div class="quick-actions">
        <h2 class="section-title">Quick Actions</h2>
        <div class="actions-grid">
            <a href="<?php echo ADMIN_URL; ?>/modules/add-level.php" class="action-card">
                <div class="action-icon">
                    <span class="icon">➕</span>
                </div>
                <div class="action-content">
                    <h3 class="action-title">Add New Level</h3>
                    <p class="action-description">Create a new game level</p>
                </div>
            </a>
            
            <a href="<?php echo ADMIN_URL; ?>/modules/users.php" class="action-card">
                <div class="action-icon">
                    <span class="icon">👥</span>
                </div>
                <div class="action-content">
                    <h3 class="action-title">Manage Users</h3>
                    <p class="action-description">View and manage game users</p>
                </div>
            </a>
            
            <a href="<?php echo ADMIN_URL; ?>/modules/levels.php" class="action-card">
                <div class="action-icon">
                    <span class="icon">🎮</span>
                </div>
                <div class="action-content">
                    <h3 class="action-title">Manage Levels</h3>
                    <p class="action-description">Edit and organize game levels</p>
                </div>
            </a>
            
            <a href="<?php echo ADMIN_URL; ?>/modules/settings.php" class="action-card">
                <div class="action-icon">
                    <span class="icon">⚙️</span>
                </div>
                <div class="action-content">
                    <h3 class="action-title">Settings</h3>
                    <p class="action-description">Configure game settings</p>
                </div>
            </a>
        </div>
    </div>
</div>

<?php
// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
