<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Analytics';

// Set current page for navigation
$current_page = 'analytics.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Include header
include_once ADMIN_PATH . '/includes/header.php';

// Get analytics data
try {
    $db = Database::getInstance();
    
    // Total users
    $totalUsers = fetch("SELECT COUNT(*) as total FROM users")['total'];
    
    // Total plays
    $totalPlays = fetch("SELECT COUNT(*) as total FROM game_plays")['total'];
    
    // Daily active users (last 7 days)
    $dailyActiveUsers = fetch("
        SELECT COUNT(DISTINCT user_id) as daily_users 
        FROM game_plays 
        WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
    ")['daily_users'];
    
    // Most played level
    $mostPlayedLevel = fetch("
        SELECT l.level_number, l.name, COUNT(gp.id) as play_count
        FROM levels l
        LEFT JOIN game_plays gp ON l.id = gp.level_id
        GROUP BY l.id, l.level_number, l.name
        ORDER BY play_count DESC
        LIMIT 1
    ");
    
    // User registration trend (last 30 days)
    $registrationTrend = fetchAll("
        SELECT DATE(created_at) as date, COUNT(*) as registrations
        FROM users
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        ORDER BY date ASC
    ");
    
    // Game plays trend (last 30 days)
    $playsTrend = fetchAll("
        SELECT DATE(created_at) as date, COUNT(*) as plays
        FROM game_plays
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        ORDER BY date ASC
    ");
    
    // Level difficulty distribution
    $difficultyStats = fetchAll("
        SELECT difficulty, COUNT(*) as count
        FROM levels
        GROUP BY difficulty
        ORDER BY FIELD(difficulty, 'Easy', 'Medium', 'Hard', 'Expert')
    ");
    
    // Top 5 most active users
    $topUsers = fetchAll("
        SELECT u.username, u.games_played, u.games_won, u.coins
        FROM users u
        ORDER BY u.games_played DESC
        LIMIT 5
    ");
    
    // Completion rate by difficulty
    $completionByDifficulty = fetchAll("
        SELECT 
            l.difficulty,
            COUNT(gp.id) as total_plays,
            SUM(CASE WHEN gp.status = 'won' THEN 1 ELSE 0 END) as completed,
            ROUND(SUM(CASE WHEN gp.status = 'won' THEN 1 ELSE 0 END) / COUNT(gp.id) * 100, 1) as completion_rate
        FROM levels l
        LEFT JOIN game_plays gp ON l.id = gp.level_id
        GROUP BY l.difficulty
        ORDER BY FIELD(l.difficulty, 'Easy', 'Medium', 'Hard', 'Expert')
    ");
    
    // Revenue analytics (if applicable)
    $totalRevenue = fetch("SELECT COALESCE(SUM(amount), 0) as total FROM purchases WHERE status = 'completed'")['total'];
    $revenueToday = fetch("
        SELECT COALESCE(SUM(amount), 0) as total 
        FROM purchases 
        WHERE status = 'completed' AND DATE(created_at) = CURDATE()
    ")['total'];
    
} catch (Exception $e) {
    error_log("Analytics error: " . $e->getMessage());
    $totalUsers = $totalPlays = $dailyActiveUsers = 0;
    $mostPlayedLevel = null;
    $registrationTrend = $playsTrend = $difficultyStats = $topUsers = $completionByDifficulty = [];
    $totalRevenue = $revenueToday = 0;
}
?>

<!-- Analytics Content -->
<div class="analytics-content">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Analytics Dashboard</h1>
        <p class="page-description">Comprehensive insights into game performance and user engagement</p>
    </div>
    
    <!-- Key Metrics -->
    <div class="metrics-grid">
        <div class="metric-card">
            <div class="metric-icon">
                <span class="icon">👥</span>
            </div>
            <div class="metric-content">
                <h3 class="metric-title">Total Users</h3>
                <p class="metric-value"><?php echo number_format($totalUsers); ?></p>
                <div class="metric-change">
                    <span class="change-text">Registered players</span>
                </div>
            </div>
        </div>
        
        <div class="metric-card">
            <div class="metric-icon">
                <span class="icon">🎮</span>
            </div>
            <div class="metric-content">
                <h3 class="metric-title">Total Plays</h3>
                <p class="metric-value"><?php echo number_format($totalPlays); ?></p>
                <div class="metric-change">
                    <span class="change-text">All time game sessions</span>
                </div>
            </div>
        </div>
        
        <div class="metric-card">
            <div class="metric-icon">
                <span class="icon">📅</span>
            </div>
            <div class="metric-content">
                <h3 class="metric-title">Daily Active Users</h3>
                <p class="metric-value"><?php echo number_format($dailyActiveUsers); ?></p>
                <div class="metric-change">
                    <span class="change-text">Last 7 days</span>
                </div>
            </div>
        </div>
        
        <div class="metric-card">
            <div class="metric-icon">
                <span class="icon">🏆</span>
            </div>
            <div class="metric-content">
                <h3 class="metric-title">Most Played Level</h3>
                <p class="metric-value">
                    <?php 
                    if ($mostPlayedLevel && $mostPlayedLevel['play_count'] > 0) {
                        echo '#' . $mostPlayedLevel['level_number'];
                    } else {
                        echo 'N/A';
                    }
                    ?>
                </p>
                <div class="metric-change">
                    <span class="change-text">
                        <?php 
                        if ($mostPlayedLevel && $mostPlayedLevel['play_count'] > 0) {
                            echo $mostPlayedLevel['play_count'] . ' plays';
                        } else {
                            echo 'No data';
                        }
                        ?>
                    </span>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Analytics Grid -->
    <div class="analytics-grid">
        <!-- User Registration Trend -->
        <div class="analytics-card">
            <div class="card-header">
                <h2 class="card-title">User Registration Trend (30 Days)</h2>
                <div class="card-actions">
                    <button class="btn btn-sm btn-secondary" onclick="refreshAnalytics()">
                        <span class="btn-icon">🔄</span>
                        Refresh
                    </button>
                </div>
            </div>
            <div class="card-content">
                <?php if (empty($registrationTrend)): ?>
                <div class="empty-state">
                    <p>No registration data available</p>
                </div>
                <?php else: ?>
                <div class="chart-container">
                    <canvas id="registrationChart" width="400" height="200"></canvas>
                </div>
                <div class="chart-summary">
                    <div class="summary-item">
                        <span class="summary-label">Total New Users:</span>
                        <span class="summary-value"><?php echo array_sum(array_column($registrationTrend, 'registrations')); ?></span>
                    </div>
                    <div class="summary-item">
                        <span class="summary-label">Daily Average:</span>
                        <span class="summary-value"><?php echo round(array_sum(array_column($registrationTrend, 'registrations')) / max(1, count($registrationTrend)), 1); ?></span>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Game Plays Trend -->
        <div class="analytics-card">
            <div class="card-header">
                <h2 class="card-title">Game Plays Trend (30 Days)</h2>
                <div class="card-actions">
                    <select class="filter-select" onchange="filterPlays(this.value)">
                        <option value="30">Last 30 Days</option>
                        <option value="7">Last 7 Days</option>
                        <option value="1">Today</option>
                    </select>
                </div>
            </div>
            <div class="card-content">
                <?php if (empty($playsTrend)): ?>
                <div class="empty-state">
                    <p>No game play data available</p>
                </div>
                <?php else: ?>
                <div class="chart-container">
                    <canvas id="playsChart" width="400" height="200"></canvas>
                </div>
                <div class="chart-summary">
                    <div class="summary-item">
                        <span class="summary-label">Total Plays:</span>
                        <span class="summary-value"><?php echo number_format(array_sum(array_column($playsTrend, 'plays'))); ?></span>
                    </div>
                    <div class="summary-item">
                        <span class="summary-label">Daily Average:</span>
                        <span class="summary-value"><?php echo round(array_sum(array_column($playsTrend, 'plays')) / max(1, count($playsTrend)), 1); ?></span>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Level Difficulty Distribution -->
        <div class="analytics-card">
            <div class="card-header">
                <h2 class="card-title">Level Difficulty Distribution</h2>
            </div>
            <div class="card-content">
                <?php if (empty($difficultyStats)): ?>
                <div class="empty-state">
                    <p>No level data available</p>
                </div>
                <?php else: ?>
                <div class="difficulty-chart">
                    <?php foreach ($difficultyStats as $stat): ?>
                    <div class="difficulty-item">
                        <div class="difficulty-label"><?php echo $stat['difficulty']; ?></div>
                        <div class="difficulty-bar">
                            <div class="difficulty-fill difficulty-<?php echo strtolower($stat['difficulty']); ?>" 
                                 style="width: <?php echo ($stat['count'] / max(1, max(array_column($difficultyStats, 'count')))) * 100; ?>%"></div>
                        </div>
                        <div class="difficulty-count"><?php echo $stat['count']; ?></div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Top Users -->
        <div class="analytics-card">
            <div class="card-header">
                <h2 class="card-title">Top 5 Most Active Users</h2>
            </div>
            <div class="card-content">
                <?php if (empty($topUsers)): ?>
                <div class="empty-state">
                    <p>No user data available</p>
                </div>
                <?php else: ?>
                <div class="top-users-list">
                    <?php foreach ($topUsers as $index => $user): ?>
                    <div class="user-rank-item">
                        <div class="rank-number"><?php echo $index + 1; ?></div>
                        <div class="user-info">
                            <div class="user-name"><?php echo htmlspecialchars($user['username']); ?></div>
                            <div class="user-stats">
                                <span class="stat"><?php echo $user['games_played']; ?> plays</span>
                                <span class="stat"><?php echo $user['games_won']; ?> wins</span>
                                <span class="stat"><?php echo number_format($user['coins']); ?> coins</span>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Completion Rate by Difficulty -->
        <div class="analytics-card">
            <div class="card-header">
                <h2 class="card-title">Completion Rate by Difficulty</h2>
            </div>
            <div class="card-content">
                <?php if (empty($completionByDifficulty)): ?>
                <div class="empty-state">
                    <p>No completion data available</p>
                </div>
                <?php else: ?>
                <div class="completion-stats">
                    <?php foreach ($completionByDifficulty as $stat): ?>
                    <div class="completion-item">
                        <div class="completion-header">
                            <span class="difficulty-name"><?php echo $stat['difficulty']; ?></span>
                            <span class="completion-rate"><?php echo $stat['completion_rate']; ?>%</span>
                        </div>
                        <div class="completion-bar">
                            <div class="completion-fill" style="width: <?php echo $stat['completion_rate']; ?>%"></div>
                        </div>
                        <div class="completion-details">
                            <span><?php echo $stat['completed']; ?> completed</span>
                            <span><?php echo $stat['total_plays']; ?> total plays</span>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Revenue Analytics -->
        <div class="analytics-card">
            <div class="card-header">
                <h2 class="card-title">Revenue Analytics</h2>
            </div>
            <div class="card-content">
                <div class="revenue-stats">
                    <div class="revenue-item">
                        <div class="revenue-label">Total Revenue</div>
                        <div class="revenue-value">$<?php echo number_format($totalRevenue, 2); ?></div>
                    </div>
                    <div class="revenue-item">
                        <div class="revenue-label">Today's Revenue</div>
                        <div class="revenue-value">$<?php echo number_format($revenueToday, 2); ?></div>
                    </div>
                </div>
                <?php if ($totalRevenue > 0): ?>
                <div class="revenue-chart">
                    <div class="revenue-bar">
                        <div class="revenue-fill" style="width: 100%"></div>
                    </div>
                    <div class="revenue-info">
                        <span>Total: $<?php echo number_format($totalRevenue, 2); ?></span>
                    </div>
                </div>
                <?php else: ?>
                <div class="empty-state">
                    <p>No revenue data available</p>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
// Chart data
const registrationData = <?php echo json_encode(array_map(function($item) {
    return ['x' => $item['date'], 'y' => $item['registrations']];
}, $registrationTrend)); ?>;

const playsData = <?php echo json_encode(array_map(function($item) {
    return ['x' => $item['date'], 'y' => $item['plays']];
}, $playsTrend)); ?>;

// Simple chart rendering (you can replace with Chart.js for better charts)
function renderSimpleChart(canvasId, data, color) {
    const canvas = document.getElementById(canvasId);
    if (!canvas || data.length === 0) return;
    
    const ctx = canvas.getContext('2d');
    const width = canvas.width;
    const height = canvas.height;
    const padding = 40;
    
    // Clear canvas
    ctx.clearRect(0, 0, width, height);
    
    // Find max value
    const maxValue = Math.max(...data.map(d => d.y));
    
    // Draw axes
    ctx.strokeStyle = '#ddd';
    ctx.beginPath();
    ctx.moveTo(padding, padding);
    ctx.lineTo(padding, height - padding);
    ctx.lineTo(width - padding, height - padding);
    ctx.stroke();
    
    // Draw data points and lines
    ctx.strokeStyle = color;
    ctx.fillStyle = color;
    ctx.beginPath();
    
    data.forEach((point, index) => {
        const x = padding + (index / (data.length - 1)) * (width - 2 * padding);
        const y = height - padding - (point.y / maxValue) * (height - 2 * padding);
        
        if (index === 0) {
            ctx.moveTo(x, y);
        } else {
            ctx.lineTo(x, y);
        }
        
        // Draw point
        ctx.fillRect(x - 2, y - 2, 4, 4);
    });
    
    ctx.stroke();
}

// Initialize charts
document.addEventListener('DOMContentLoaded', function() {
    renderSimpleChart('registrationChart', registrationData, '#667eea');
    renderSimpleChart('playsChart', playsData, '#764ba2');
});

function refreshAnalytics() {
    location.reload();
}

function filterPlays(period) {
    // This would typically make an AJAX call to get filtered data
    showNotification('Filtering by ' + period + ' days...', 'info');
    setTimeout(() => location.reload(), 1000);
}
</script>

<style>
.analytics-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 20px;
    margin-top: 30px;
}

.analytics-card {
    background: white;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    overflow: hidden;
}

.card-header {
    padding: 20px;
    border-bottom: 1px solid #f0f0f0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.card-title {
    font-size: 18px;
    font-weight: 600;
    color: #333;
}

.card-content {
    padding: 20px;
}

.chart-container {
    margin-bottom: 20px;
}

.chart-summary {
    display: flex;
    gap: 20px;
    font-size: 14px;
}

.summary-item {
    display: flex;
    flex-direction: column;
}

.summary-label {
    color: #666;
}

.summary-value {
    font-weight: 600;
    color: #333;
}

.difficulty-chart {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.difficulty-item {
    display: flex;
    align-items: center;
    gap: 15px;
}

.difficulty-label {
    min-width: 80px;
    font-weight: 500;
}

.difficulty-bar {
    flex: 1;
    height: 20px;
    background: #f0f0f0;
    border-radius: 10px;
    overflow: hidden;
}

.difficulty-fill {
    height: 100%;
    transition: width 0.3s ease;
}

.difficulty-easy { background: #4caf50; }
.difficulty-medium { background: #ff9800; }
.difficulty-hard { background: #f44336; }
.difficulty-expert { background: #9c27b0; }

.difficulty-count {
    min-width: 40px;
    text-align: right;
    font-weight: 600;
}

.top-users-list {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.user-rank-item {
    display: flex;
    align-items: center;
    gap: 15px;
}

.rank-number {
    width: 30px;
    height: 30px;
    background: #667eea;
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
}

.user-info {
    flex: 1;
}

.user-name {
    font-weight: 500;
    margin-bottom: 5px;
}

.user-stats {
    display: flex;
    gap: 15px;
    font-size: 14px;
    color: #666;
}

.completion-stats {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.completion-item {
    border: 1px solid #f0f0f0;
    border-radius: 8px;
    padding: 15px;
}

.completion-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 10px;
}

.difficulty-name {
    font-weight: 500;
}

.completion-rate {
    font-weight: 600;
    color: #4caf50;
}

.completion-bar {
    height: 8px;
    background: #f0f0f0;
    border-radius: 4px;
    overflow: hidden;
    margin-bottom: 10px;
}

.completion-fill {
    height: 100%;
    background: #4caf50;
    transition: width 0.3s ease;
}

.completion-details {
    display: flex;
    justify-content: space-between;
    font-size: 14px;
    color: #666;
}

.revenue-stats {
    display: flex;
    gap: 30px;
    margin-bottom: 20px;
}

.revenue-item {
    text-align: center;
}

.revenue-label {
    color: #666;
    margin-bottom: 5px;
}

.revenue-value {
    font-size: 24px;
    font-weight: 600;
    color: #4caf50;
}

.revenue-chart {
    margin-top: 20px;
}

.revenue-bar {
    height: 30px;
    background: #f0f0f0;
    border-radius: 15px;
    overflow: hidden;
    margin-bottom: 10px;
}

.revenue-fill {
    height: 100%;
    background: linear-gradient(90deg, #4caf50, #8bc34a);
    transition: width 0.3s ease;
}

.revenue-info {
    text-align: center;
    font-weight: 500;
}
</style>

<?php
// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
