<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Set page title
$page_title = 'Add Level';

// Set current page for navigation
$current_page = 'add-level.php';

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Include header
include_once ADMIN_PATH . '/includes/header.php';

// Initialize variables
$errors = [];
$success = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $errors[] = 'Invalid request token';
    } else {
        // Sanitize and validate inputs
        $levelNumber = intval($_POST['level_number'] ?? 0);
        $name = sanitizeInput($_POST['name'] ?? '');
        $description = sanitizeInput($_POST['description'] ?? '');
        $difficulty = sanitizeInput($_POST['difficulty'] ?? 'Easy');
        $bottlesCount = intval($_POST['bottles_count'] ?? 4);
        $colorsCount = intval($_POST['colors_count'] ?? 2);
        $bottleCapacity = intval($_POST['bottle_capacity'] ?? 4);
        $solutionMoves = intval($_POST['solution_moves'] ?? 0);
        $timeLimit = intval($_POST['time_limit'] ?? 0);
        $rewardCoins = intval($_POST['reward_coins'] ?? 10);
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        
        // Validate required fields
        if (empty($name)) {
            $errors['name'] = 'Level name is required';
        }
        
        if ($levelNumber <= 0) {
            $errors['level_number'] = 'Level number must be greater than 0';
        }
        
        if ($bottlesCount < 3 || $bottlesCount > 20) {
            $errors['bottles_count'] = 'Bottles count must be between 3 and 20';
        }
        
        if ($colorsCount < 2 || $colorsCount > 10) {
            $errors['colors_count'] = 'Colors count must be between 2 and 10';
        }
        
        if ($bottleCapacity < 3 || $bottleCapacity > 10) {
            $errors['bottle_capacity'] = 'Bottle capacity must be between 3 and 10';
        }
        
        if (!in_array($difficulty, ['Easy', 'Medium', 'Hard', 'Expert'])) {
            $errors['difficulty'] = 'Invalid difficulty level';
        }
        
        // Validate initial state JSON
        $initialState = $_POST['initial_state'] ?? '';
        if (empty($initialState)) {
            $errors['initial_state'] = 'Initial state is required';
        } else {
            $decoded = json_decode($initialState, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $errors['initial_state'] = 'Invalid JSON format: ' . json_last_error_msg();
            } elseif (!is_array($decoded) || empty($decoded)) {
                $errors['initial_state'] = 'Initial state must be a non-empty array';
            }
        }
        
        // Check if level number already exists
        if (empty($errors)) {
            $existingLevel = fetch("SELECT id FROM levels WHERE level_number = ?", [$levelNumber]);
            if ($existingLevel) {
                $errors['level_number'] = 'Level number already exists';
            }
        }
        
        // If no errors, save the level
        if (empty($errors)) {
            try {
                $db = Database::getInstance();
                $db->beginTransaction();
                
                // Insert level
                $levelData = [
                    'level_number' => $levelNumber,
                    'name' => $name,
                    'description' => $description,
                    'difficulty' => $difficulty,
                    'bottles_count' => $bottlesCount,
                    'colors_count' => $colorsCount,
                    'bottle_capacity' => $bottleCapacity,
                    'initial_state' => $initialState,
                    'solution_moves' => $solutionMoves,
                    'time_limit' => $timeLimit,
                    'reward_coins' => $rewardCoins,
                    'is_active' => $isActive
                ];
                
                $levelId = insert('levels', $levelData);
                
                // Log activity
                logActivity('add_level', 'levels', $levelId, null, $levelData);
                
                $db->commit();
                
                $success = 'Level created successfully! <a href="' . ADMIN_URL . '/modules/levels.php">View all levels</a> or <a href="' . ADMIN_URL . '/modules/add-level.php">Create another level</a>';
                
                // Clear form
                $_POST = [];
                
            } catch (Exception $e) {
                $db->rollBack();
                $errors[] = 'Error saving level: ' . $e->getMessage();
            }
        }
    }
}

// Get next available level number
$nextLevelNumber = 1;
$lastLevel = fetch("SELECT MAX(level_number) as max_level FROM levels");
if ($lastLevel && $lastLevel['max_level']) {
    $nextLevelNumber = $lastLevel['max_level'] + 1;
}

// Generate sample level data
function generateSampleLevel($bottles, $colors, $capacity) {
    $colorNames = ['red', 'blue', 'green', 'yellow', 'purple', 'orange'];
    $selectedColors = array_slice($colorNames, 0, $colors);
    
    $level = [];
    
    // Fill bottles with colors
    for ($i = 0; $i < $bottles - 2; $i++) {
        $bottle = [];
        for ($j = 0; $j < $capacity; $j++) {
            $bottle[] = $selectedColors[$i % $colors];
        }
        $level[] = $bottle;
    }
    
    // Add empty bottles
    $level[] = [];
    $level[] = [];
    
    return json_encode($level);
}
?>

<!-- Add Level Content -->
<div class="add-level-content">
    <!-- Display Notifications -->
    <?php displayNotifications(); ?>
    
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Add New Level</h1>
        <p class="page-description">Create a new game level for the Water Sort Game</p>
    </div>
    
    <!-- Success Message -->
    <?php if (!empty($success)): ?>
    <div class="alert alert-success">
        <span class="alert-icon">✅</span>
        <span class="alert-text"><?php echo $success; ?></span>
    </div>
    <?php endif; ?>
    
    <!-- Add Level Form -->
    <div class="form-container">
        <form method="POST" class="level-form" data-validate>
            <!-- CSRF Token -->
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <!-- Basic Information -->
            <div class="form-section">
                <h3 class="section-title">Basic Information</h3>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="level_number" class="form-label">Level Number *</label>
                        <input type="number" 
                               id="level_number" 
                               name="level_number" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['level_number'] ?? $nextLevelNumber); ?>"
                               min="1"
                               required>
                        <?php if (isset($errors['level_number'])): ?>
                        <div class="field-error"><?php echo $errors['level_number']; ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="name" class="form-label">Level Name *</label>
                        <input type="text" 
                               id="name" 
                               name="name" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>"
                               placeholder="Enter level name"
                               required>
                        <?php if (isset($errors['name'])): ?>
                        <div class="field-error"><?php echo $errors['name']; ?></div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="description" class="form-label">Description</label>
                    <textarea id="description" 
                              name="description" 
                              class="form-textarea" 
                              placeholder="Enter level description"
                              rows="3"><?php echo htmlspecialchars($_POST['description'] ?? ''); ?></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="difficulty" class="form-label">Difficulty *</label>
                        <select id="difficulty" name="difficulty" class="form-select" required>
                            <option value="Easy" <?php echo (($_POST['difficulty'] ?? 'Easy') === 'Easy') ? 'selected' : ''; ?>>Easy</option>
                            <option value="Medium" <?php echo (($_POST['difficulty'] ?? '') === 'Medium') ? 'selected' : ''; ?>>Medium</option>
                            <option value="Hard" <?php echo (($_POST['difficulty'] ?? '') === 'Hard') ? 'selected' : ''; ?>>Hard</option>
                            <option value="Expert" <?php echo (($_POST['difficulty'] ?? '') === 'Expert') ? 'selected' : ''; ?>>Expert</option>
                        </select>
                        <?php if (isset($errors['difficulty'])): ?>
                        <div class="field-error"><?php echo $errors['difficulty']; ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="reward_coins" class="form-label">Reward Coins</label>
                        <input type="number" 
                               id="reward_coins" 
                               name="reward_coins" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['reward_coins'] ?? 10); ?>"
                               min="0"
                               max="999">
                    </div>
                </div>
            </div>
            
            <!-- Level Configuration -->
            <div class="form-section">
                <h3 class="section-title">Level Configuration</h3>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="bottles_count" class="form-label">Number of Bottles *</label>
                        <input type="number" 
                               id="bottles_count" 
                               name="bottles_count" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['bottles_count'] ?? 4); ?>"
                               min="3"
                               max="20"
                               required>
                        <?php if (isset($errors['bottles_count'])): ?>
                        <div class="field-error"><?php echo $errors['bottles_count']; ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="colors_count" class="form-label">Number of Colors *</label>
                        <input type="number" 
                               id="colors_count" 
                               name="colors_count" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['colors_count'] ?? 2); ?>"
                               min="2"
                               max="10"
                               required>
                        <?php if (isset($errors['colors_count'])): ?>
                        <div class="field-error"><?php echo $errors['colors_count']; ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="bottle_capacity" class="form-label">Bottle Capacity *</label>
                        <input type="number" 
                               id="bottle_capacity" 
                               name="bottle_capacity" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['bottle_capacity'] ?? 4); ?>"
                               min="3"
                               max="10"
                               required>
                        <?php if (isset($errors['bottle_capacity'])): ?>
                        <div class="field-error"><?php echo $errors['bottle_capacity']; ?></div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="solution_moves" class="form-label">Solution Moves (Optional)</label>
                        <input type="number" 
                               id="solution_moves" 
                               name="solution_moves" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['solution_moves'] ?? 0); ?>"
                               min="0"
                               placeholder="Minimum moves to solve">
                    </div>
                    
                    <div class="form-group">
                        <label for="time_limit" class="form-label">Time Limit (Optional)</label>
                        <input type="number" 
                               id="time_limit" 
                               name="time_limit" 
                               class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['time_limit'] ?? 0); ?>"
                               min="0"
                               placeholder="Seconds (0 = no limit)">
                    </div>
                </div>
            </div>
            
            <!-- Initial State -->
            <div class="form-section">
                <h3 class="section-title">Initial State (JSON) *</h3>
                <p class="section-description">Define the initial state of bottles as a JSON array. Each bottle is an array of colors.</p>
                
                <div class="form-group">
                    <label for="initial_state" class="form-label">Initial State JSON *</label>
                    <textarea id="initial_state" 
                              name="initial_state" 
                              class="form-textarea code-textarea" 
                              placeholder='[["red","red","blue","blue"], ["blue","red","blue","red"], [], []]'
                              rows="8"
                              required><?php echo htmlspecialchars($_POST['initial_state'] ?? generateSampleLevel(4, 2, 4)); ?></textarea>
                    <?php if (isset($errors['initial_state'])): ?>
                    <div class="field-error"><?php echo $errors['initial_state']; ?></div>
                    <?php endif; ?>
                </div>
                
                <div class="json-tools">
                    <button type="button" class="btn btn-secondary" onclick="validateJSON()">
                        <span class="btn-icon">✅</span>
                        Validate JSON
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="generateSample()">
                        <span class="btn-icon">🎲</span>
                        Generate Sample
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="formatJSON()">
                        <span class="btn-icon">🎨</span>
                        Format JSON
                    </button>
                </div>
            </div>
            
            <!-- Status -->
            <div class="form-section">
                <h3 class="section-title">Status</h3>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="is_active" class="checkbox-input" <?php echo isset($_POST['is_active']) ? 'checked' : ''; ?>>
                        <span class="checkbox-text">Level is active (available for players)</span>
                    </label>
                </div>
            </div>
            
            <!-- Form Actions -->
            <div class="form-actions">
                <button type="submit" class="btn btn-primary">
                    <span class="btn-icon">💾</span>
                    Create Level
                </button>
                <a href="<?php echo ADMIN_URL; ?>/modules/levels.php" class="btn btn-secondary">
                    <span class="btn-icon">✖</span>
                    Cancel
                </a>
            </div>
        </form>
    </div>
</div>

<script>
function validateJSON() {
    const textarea = document.getElementById('initial_state');
    const value = textarea.value;
    
    try {
        const parsed = JSON.parse(value);
        if (!Array.isArray(parsed)) {
            throw new Error('JSON must be an array');
        }
        showNotification('JSON is valid! Found ' + parsed.length + ' bottles.', 'success');
    } catch (e) {
        showNotification('Invalid JSON: ' + e.message, 'error');
    }
}

function generateSample() {
    const bottles = parseInt(document.getElementById('bottles_count').value) || 4;
    const colors = parseInt(document.getElementById('colors_count').value) || 2;
    const capacity = parseInt(document.getElementById('bottle_capacity').value) || 4;
    
    const colorNames = ['red', 'blue', 'green', 'yellow', 'purple', 'orange'];
    const selectedColors = colorNames.slice(0, colors);
    
    let level = [];
    
    // Fill bottles with colors
    for (let i = 0; i < bottles - 2; i++) {
        let bottle = [];
        for (let j = 0; j < capacity; j++) {
            bottle.push(selectedColors[i % colors]);
        }
        level.push(bottle);
    }
    
    // Add empty bottles
    level.push([]);
    level.push([]);
    
    document.getElementById('initial_state').value = JSON.stringify(level, null, 2);
    showNotification('Sample level generated!', 'success');
}

function formatJSON() {
    const textarea = document.getElementById('initial_state');
    const value = textarea.value;
    
    try {
        const parsed = JSON.parse(value);
        textarea.value = JSON.stringify(parsed, null, 2);
        showNotification('JSON formatted successfully!', 'success');
    } catch (e) {
        showNotification('Cannot format invalid JSON', 'error');
    }
}

// Auto-generate sample when configuration changes
document.getElementById('bottles_count').addEventListener('change', generateSample);
document.getElementById('colors_count').addEventListener('change', generateSample);
document.getElementById('bottle_capacity').addEventListener('change', generateSample);
</script>

<?php
// Include footer
include_once ADMIN_PATH . '/includes/footer.php';
?>
