<?php
/**
 * Helper Functions
 * Water Sort Game Admin Panel
 */

/**
 * Security Functions
 */

/**
 * Generate CSRF token
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(CSRF_TOKEN_LENGTH));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF token
 */
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Validate CSRF token (alias for verifyCSRFToken)
 */
function validateCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Show notification message
 */
function showNotification($message, $type = 'info') {
    // Store notification in session for display on next page load
    $_SESSION['notification'] = [
        'message' => $message,
        'type' => $type
    ];
}

/**
 * Display any stored notifications
 */
function displayNotifications() {
    if (isset($_SESSION['notification'])) {
        $notification = $_SESSION['notification'];
        echo '<div class="alert alert-' . $notification['type'] . '">';
        echo '<span class="alert-icon">' . getNotificationIcon($notification['type']) . '</span>';
        echo '<span class="alert-text">' . htmlspecialchars($notification['message']) . '</span>';
        echo '</div>';
        unset($_SESSION['notification']);
    }
}

/**
 * Get notification icon based on type
 */
function getNotificationIcon($type) {
    $icons = [
        'success' => '✅',
        'error' => '❌',
        'warning' => '⚠️',
        'info' => 'ℹ️'
    ];
    return $icons[$type] ?? 'ℹ️';
}

/**
 * Sanitize input data
 */
function sanitize($input) {
    if (is_array($input)) {
        return array_map('sanitize', $input);
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Validate email
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Hash password securely
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_ARGON2ID);
}

/**
 * Verify password
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Generate secure random string
 */
function generateRandomString($length = 32) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Generate random token
 */
function generateRandomToken($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * Check if request is AJAX
 */
function isAjaxRequest() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * Get client IP address
 */
function getClientIP() {
    $ipKeys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'];
    
    foreach ($ipKeys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
}

/**
 * String Functions
 */

/**
 * Limit string length
 */
function limitString($string, $length = 50, $suffix = '...') {
    if (strlen($string) <= $length) {
        return $string;
    }
    return substr($string, 0, $length - strlen($suffix)) . $suffix;
}

/**
 * Convert to title case
 */
function toTitleCase($string) {
    return ucwords(strtolower($string));
}

/**
 * Generate slug from string
 */
function generateSlug($string) {
    $slug = strtolower($string);
    $slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
    return trim($slug, '-');
}

/**
 * Date/Time Functions
 */

/**
 * Format date/time
 */
function formatDate($date, $format = 'M j, Y g:i A') {
    if (!$date || $date === '0000-00-00 00:00:00') {
        return 'Never';
    }
    return date($format, strtotime($date));
}

/**
 * Get time ago string
 */
function timeAgo($datetime) {
    if (!$datetime) {
        return 'Never';
    }
    
    $time = strtotime($datetime);
    $now = time();
    $diff = $now - $time;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        return round($diff / 60) . ' minutes ago';
    } elseif ($diff < 86400) {
        return round($diff / 3600) . ' hours ago';
    } elseif ($diff < 2592000) {
        return round($diff / 86400) . ' days ago';
    } else {
        return formatDate($datetime);
    }
}

/**
 * Number Functions
 */

/**
 * Format number with thousands separator
 */
function formatNumber($number) {
    return number_format($number, 0, '.', ',');
}

/**
 * Format currency
 */
function formatCurrency($amount, $currency = '$') {
    return $currency . number_format($amount, 2);
}

/**
 * Format bytes to human readable
 */
function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

/**
 * Array Functions
 */

/**
 * Get array value safely
 */
function arrayGet($array, $key, $default = null) {
    return isset($array[$key]) ? $array[$key] : $default;
}

/**
 * Check if array is associative
 */
function isAssociativeArray($array) {
    if (!is_array($array) || empty($array)) {
        return false;
    }
    return array_keys($array) !== range(0, count($array) - 1);
}

/**
 * File Functions
 */

/**
 * Get file extension
 */
function getFileExtension($filename) {
    return strtolower(pathinfo($filename, PATHINFO_EXTENSION));
}

/**
 * Generate unique filename
 */
function generateUniqueFilename($filename) {
    $extension = getFileExtension($filename);
    $basename = pathinfo($filename, PATHINFO_FILENAME);
    return $basename . '_' . time() . '_' . generateRandomString(8) . '.' . $extension;
}

/**
 * URL Functions
 */

/**
 * Get current URL
 */
function getCurrentURL() {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    return $protocol . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
}

/**
 * Build URL with parameters
 */
function buildURL($base, $params = []) {
    $url = $base;
    if (!empty($params)) {
        $url .= '?' . http_build_query($params);
    }
    return $url;
}

/**
 * Redirect to URL
 */
function redirect($url, $statusCode = 302) {
    if (!headers_sent()) {
        header("Location: $url", true, $statusCode);
        exit;
    } else {
        echo "<script>window.location.href='$url';</script>";
        exit;
    }
}

/**
 * Validation Functions
 */

/**
 * Validate required fields
 */
function validateRequired($data, $required) {
    $errors = [];
    foreach ($required as $field) {
        if (!isset($data[$field]) || empty(trim($data[$field]))) {
            $errors[$field] = ucfirst(str_replace('_', ' ', $field)) . ' is required';
        }
    }
    return $errors;
}

/**
 * Validate numeric value
 */
function validateNumeric($value, $min = null, $max = null) {
    if (!is_numeric($value)) {
        return false;
    }
    
    $value = (float) $value;
    
    if ($min !== null && $value < $min) {
        return false;
    }
    
    if ($max !== null && $value > $max) {
        return false;
    }
    
    return true;
}

/**
 * Validate string length
 */
function validateLength($value, $min = null, $max = null) {
    $length = strlen($value);
    
    if ($min !== null && $length < $min) {
        return false;
    }
    
    if ($max !== null && $length > $max) {
        return false;
    }
    
    return true;
}

/**
 * JSON Functions
 */

/**
 * Safe JSON encode
 */
function safeJSONEncode($data) {
    $json = json_encode($data, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP);
    return $json === false ? false : $json;
}

/**
 * Safe JSON decode
 */
function safeJSONDecode($json, $assoc = true) {
    $data = json_decode($json, $assoc);
    return json_last_error() === JSON_ERROR_NONE ? $data : false;
}

/**
 * Debug Functions
 */

/**
 * Debug print (only in debug mode)
 */
function debug($var, $label = 'DEBUG') {
    if (DEBUG_MODE) {
        echo "<pre><strong>$label:</strong><br>";
        print_r($var);
        echo "</pre>";
    }
}

/**
 * Log to file
 */
function logMessage($message, $level = 'INFO') {
    $logFile = __DIR__ . '/../logs/app.log';
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message" . PHP_EOL;
    
    $dir = dirname($logFile);
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
    
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

/**
 * Additional helper functions for security
 */
function sanitizeInput($input) {
    return sanitize($input);
}

function formatDateTime($date) {
    return formatDate($date);
}
?>
