<?php
// Define admin access for security
define('ADMIN_ACCESS', true);

// Include required files
require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/functions.php';
require_once __DIR__ . '/../includes/security.php';

// Initialize variables
$error = '';
$success = '';
$timeout = false;

// Check if user is already logged in
if (Auth::isLoggedIn()) {
    header('Location: ' . ADMIN_URL . '/modules/dashboard.php');
    exit;
}

// Check for session timeout
if (isset($_GET['timeout']) && $_GET['timeout'] == '1') {
    $timeout = true;
}

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please try again.';
    } else {
        $username = sanitizeInput($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        $remember = isset($_POST['remember']);
        
        // Validate input
        if (empty($username) || empty($password)) {
            $error = 'Please enter both username and password.';
        } else {
            // Attempt login
            $loginResult = Auth::login($username, $password, $remember);
            
            if ($loginResult['success']) {
                // Redirect to dashboard
                header('Location: ' . ADMIN_URL . '/modules/dashboard.php');
                exit;
            } else {
                $error = $loginResult['message'];
            }
        }
    }
}

// Generate CSRF token
$csrf_token = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="X-UA-Compatible" content="ie=edge">
    <title>Login - Water Sort Game Admin</title>
    
    <!-- CSS -->
    <link rel="stylesheet" href="<?php echo ADMIN_URL; ?>/assets/css/admin.css">
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="<?php echo ADMIN_URL; ?>/assets/favicon.ico">
</head>
<body class="login-page">
    <div class="login-container">
        <div class="login-card">
            <!-- Logo and Title -->
            <div class="login-header">
                <div class="logo">
                    <span class="logo-icon">💧</span>
                    <h1>Water Sort Game</h1>
                </div>
                <p class="subtitle">Admin Panel Login</p>
            </div>
            
            <!-- Login Form -->
            <form method="POST" action="" class="login-form" data-validate>
                <!-- CSRF Token -->
                <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                
                <!-- Error/Success Messages -->
                <?php if ($error): ?>
                <div class="alert alert-error">
                    <span class="alert-icon">⚠️</span>
                    <span class="alert-text"><?php echo htmlspecialchars($error); ?></span>
                </div>
                <?php endif; ?>
                
                <?php if ($timeout): ?>
                <div class="alert alert-warning">
                    <span class="alert-icon">⏰</span>
                    <span class="alert-text">Your session has expired. Please login again.</span>
                </div>
                <?php endif; ?>
                
                <?php if ($success): ?>
                <div class="alert alert-success">
                    <span class="alert-icon">✅</span>
                    <span class="alert-text"><?php echo htmlspecialchars($success); ?></span>
                </div>
                <?php endif; ?>
                
                <!-- Username Field -->
                <div class="form-group">
                    <label for="username" class="form-label">Username</label>
                    <div class="input-group">
                        <span class="input-icon">👤</span>
                        <input type="text" 
                               id="username" 
                               name="username" 
                               class="form-input" 
                               placeholder="Enter your username"
                               required
                               autocomplete="username"
                               autofocus>
                    </div>
                </div>
                
                <!-- Password Field -->
                <div class="form-group">
                    <label for="password" class="form-label">Password</label>
                    <div class="input-group">
                        <span class="input-icon">🔒</span>
                        <input type="password" 
                               id="password" 
                               name="password" 
                               class="form-input" 
                               placeholder="Enter your password"
                               required
                               autocomplete="current-password">
                        <button type="button" class="password-toggle" title="Toggle password visibility">
                            <span class="toggle-icon">👁️</span>
                        </button>
                    </div>
                </div>
                
                <!-- Remember Me -->
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="remember" class="checkbox-input">
                        <span class="checkbox-text">Remember me for 30 days</span>
                    </label>
                </div>
                
                <!-- Submit Button -->
                <button type="submit" class="btn btn-primary btn-full">
                    <span class="btn-text">Login</span>
                    <span class="btn-icon">🚀</span>
                </button>
            </form>
            
            <!-- Footer Info -->
            <div class="login-footer">
                <div class="footer-links">
                    <a href="#" class="footer-link">Forgot Password?</a>
                    <a href="#" class="footer-link">Need Help?</a>
                </div>
                <div class="footer-info">
                    <p>Default credentials: admin / admin123</p>
                    <p class="version">Version 1.0.0</p>
                </div>
            </div>
        </div>
        
        <!-- Background Decoration -->
        <div class="login-background">
            <div class="bubble bubble-1"></div>
            <div class="bubble bubble-2"></div>
            <div class="bubble bubble-3"></div>
            <div class="bubble bubble-4"></div>
            <div class="bubble bubble-5"></div>
        </div>
    </div>
    
    <!-- JavaScript -->
    <script>
        // Password toggle functionality
        document.querySelector('.password-toggle').addEventListener('click', function() {
            const passwordInput = document.getElementById('password');
            const toggleIcon = this.querySelector('.toggle-icon');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                toggleIcon.textContent = '🙈';
            } else {
                passwordInput.type = 'password';
                toggleIcon.textContent = '👁️';
            }
        });
        
        // Form validation
        document.querySelector('.login-form').addEventListener('submit', function(e) {
            const username = document.getElementById('username').value.trim();
            const password = document.getElementById('password').value;
            
            if (!username || !password) {
                e.preventDefault();
                alert('Please enter both username and password.');
                return false;
            }
        });
        
        // Auto-focus username field
        document.getElementById('username').focus();
        
        // Add loading state to submit button
        document.querySelector('.login-form').addEventListener('submit', function() {
            const submitBtn = this.querySelector('button[type="submit"]');
            const btnText = submitBtn.querySelector('.btn-text');
            const btnIcon = submitBtn.querySelector('.btn-icon');
            
            submitBtn.disabled = true;
            btnText.textContent = 'Logging in...';
            btnIcon.textContent = '⏳';
        });
        
        // Handle Enter key in password field
        document.getElementById('password').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                document.querySelector('.login-form').submit();
            }
        });
        
        // Animate background bubbles
        function animateBubbles() {
            const bubbles = document.querySelectorAll('.bubble');
            bubbles.forEach((bubble, index) => {
                const delay = index * 0.5;
                bubble.style.animationDelay = `${delay}s`;
            });
        }
        
        animateBubbles();
        
        // Check for login attempts and show warning after multiple failures
        let loginAttempts = parseInt(localStorage.getItem('loginAttempts') || '0');
        const maxAttempts = 5;
        
        if (loginAttempts >= maxAttempts) {
            const warningDiv = document.createElement('div');
            warningDiv.className = 'alert alert-warning';
            warningDiv.innerHTML = `
                <span class="alert-icon">⚠️</span>
                <span class="alert-text">
                    Multiple failed login attempts detected. 
                    Please wait 5 minutes before trying again or contact support.
                </span>
            `;
            
            const form = document.querySelector('.login-form');
            form.parentNode.insertBefore(warningDiv, form);
            
            // Disable form temporarily
            form.style.opacity = '0.5';
            form.style.pointerEvents = 'none';
            
            setTimeout(() => {
                form.style.opacity = '1';
                form.style.pointerEvents = 'auto';
                warningDiv.remove();
                localStorage.removeItem('loginAttempts');
            }, 5 * 60 * 1000); // 5 minutes
        }
        
        // Track failed attempts
        <?php if ($error): ?>
        loginAttempts++;
        localStorage.setItem('loginAttempts', loginAttempts.toString());
        <?php else: ?>
        localStorage.removeItem('loginAttempts');
        <?php endif; ?>
    </script>
</body>
</html>
