/**
 * Water Sort Game Admin Panel JavaScript
 * Handles interactive functionality for the admin panel
 */

// Global variables
let csrfToken = document.querySelector('meta[name="csrf-token"]')?.content;

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    initializeAdminPanel();
});

/**
 * Initialize admin panel functionality
 */
function initializeAdminPanel() {
    // Initialize tooltips
    initializeTooltips();
    
    // Initialize confirm dialogs
    initializeConfirmDialogs();
    
    // Initialize form validation
    initializeFormValidation();
    
    // Initialize AJAX forms
    initializeAjaxForms();
    
    // Initialize data tables
    initializeDataTables();
    
    // Initialize notifications
    initializeNotifications();
    
    // Initialize search functionality
    initializeSearch();
    
    // Initialize modal dialogs
    initializeModals();
    
    // Initialize file uploads
    initializeFileUploads();
    
    // Initialize date/time pickers
    initializeDatePickers();
    
    // Initialize auto-refresh
    initializeAutoRefresh();
}

/**
 * Initialize tooltips
 */
function initializeTooltips() {
    const tooltipElements = document.querySelectorAll('[title]');
    
    tooltipElements.forEach(element => {
        element.addEventListener('mouseenter', function(e) {
            showTooltip(e.target, e.target.getAttribute('title'));
        });
        
        element.addEventListener('mouseleave', function() {
            hideTooltip();
        });
        
        // Remove title to prevent default tooltip
        element.setAttribute('data-title', element.getAttribute('title'));
        element.removeAttribute('title');
    });
}

/**
 * Show tooltip
 */
function showTooltip(element, text) {
    // Remove existing tooltip
    hideTooltip();
    
    const tooltip = document.createElement('div');
    tooltip.className = 'tooltip';
    tooltip.textContent = text;
    document.body.appendChild(tooltip);
    
    const rect = element.getBoundingClientRect();
    tooltip.style.left = rect.left + (rect.width / 2) - (tooltip.offsetWidth / 2) + 'px';
    tooltip.style.top = rect.top - tooltip.offsetHeight - 10 + 'px';
    
    // Add animation
    setTimeout(() => tooltip.classList.add('show'), 10);
}

/**
 * Hide tooltip
 */
function hideTooltip() {
    const tooltip = document.querySelector('.tooltip');
    if (tooltip) {
        tooltip.remove();
    }
}

/**
 * Initialize confirm dialogs
 */
function initializeConfirmDialogs() {
    const confirmButtons = document.querySelectorAll('[data-confirm]');
    
    confirmButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            const message = this.getAttribute('data-confirm');
            if (!confirm(message)) {
                e.preventDefault();
                return false;
            }
        });
    });
}

/**
 * Initialize form validation
 */
function initializeFormValidation() {
    const forms = document.querySelectorAll('form[data-validate]');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!validateForm(form)) {
                e.preventDefault();
                return false;
            }
        });
    });
}

/**
 * Validate form
 */
function validateForm(form) {
    let isValid = true;
    const inputs = form.querySelectorAll('input[required], select[required], textarea[required]');
    
    inputs.forEach(input => {
        if (!input.value.trim()) {
            showFieldError(input, 'This field is required');
            isValid = false;
        } else {
            clearFieldError(input);
        }
    });
    
    return isValid;
}

/**
 * Show field error
 */
function showFieldError(field, message) {
    clearFieldError(field);
    
    const errorDiv = document.createElement('div');
    errorDiv.className = 'field-error';
    errorDiv.textContent = message;
    
    field.parentNode.appendChild(errorDiv);
    field.classList.add('error');
}

/**
 * Clear field error
 */
function clearFieldError(field) {
    const errorDiv = field.parentNode.querySelector('.field-error');
    if (errorDiv) {
        errorDiv.remove();
    }
    field.classList.remove('error');
}

/**
 * Initialize AJAX forms
 */
function initializeAjaxForms() {
    const ajaxForms = document.querySelectorAll('form[data-ajax]');
    
    ajaxForms.forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            submitAjaxForm(form);
        });
    });
}

/**
 * Submit AJAX form
 */
function submitAjaxForm(form) {
    const formData = new FormData(form);
    const submitButton = form.querySelector('button[type="submit"]');
    
    // Add CSRF token
    if (csrfToken) {
        formData.append('csrf_token', csrfToken);
    }
    
    // Disable submit button
    if (submitButton) {
        submitButton.disabled = true;
        submitButton.textContent = 'Processing...';
    }
    
    fetch(form.getAttribute('action') || window.location.href, {
        method: form.getAttribute('method') || 'POST',
        body: formData,
        credentials: 'same-origin'
    })
    .then(response => response.json())
    .then(data => {
        handleAjaxResponse(data, form);
    })
    .catch(error => {
        console.error('AJAX Error:', error);
        showNotification('An error occurred. Please try again.', 'error');
    })
    .finally(() => {
        // Re-enable submit button
        if (submitButton) {
            submitButton.disabled = false;
            submitButton.textContent = submitButton.getAttribute('data-original-text') || 'Submit';
        }
    });
}

/**
 * Handle AJAX response
 */
function handleAjaxResponse(data, form) {
    if (data.success) {
        showNotification(data.message || 'Operation completed successfully', 'success');
        
        // Reset form if needed
        if (data.reset_form) {
            form.reset();
        }
        
        // Redirect if needed
        if (data.redirect) {
            setTimeout(() => {
                window.location.href = data.redirect;
            }, 1500);
        }
        
        // Reload data if needed
        if (data.reload) {
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        }
    } else {
        showNotification(data.message || 'An error occurred', 'error');
        
        // Show field errors if any
        if (data.errors) {
            Object.keys(data.errors).forEach(fieldName => {
                const field = form.querySelector(`[name="${fieldName}"]`);
                if (field) {
                    showFieldError(field, data.errors[fieldName]);
                }
            });
        }
    }
}

/**
 * Initialize data tables
 */
function initializeDataTables() {
    const tables = document.querySelectorAll('.data-table');
    
    tables.forEach(table => {
        initializeDataTable(table);
    });
}

/**
 * Initialize data table
 */
function initializeDataTable(table) {
    const searchInput = table.parentElement.querySelector('.table-search input');
    const pagination = table.parentElement.querySelector('.table-pagination');
    
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            filterTable(table, this.value);
        });
    }
    
    if (pagination) {
        initializePagination(pagination, table);
    }
    
    // Sortable columns
    const sortableHeaders = table.querySelectorAll('th[data-sort]');
    sortableHeaders.forEach(header => {
        header.addEventListener('click', function() {
            sortTable(table, this.getAttribute('data-sort'));
        });
    });
}

/**
 * Filter table
 */
function filterTable(table, searchTerm) {
    const rows = table.querySelectorAll('tbody tr');
    const term = searchTerm.toLowerCase();
    
    rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        row.style.display = text.includes(term) ? '' : 'none';
    });
}

/**
 * Sort table
 */
function sortTable(table, column) {
    const tbody = table.querySelector('tbody');
    const rows = Array.from(tbody.querySelectorAll('tr'));
    
    rows.sort((a, b) => {
        const aValue = a.querySelector(`td[data-column="${column}"]`)?.textContent || '';
        const bValue = b.querySelector(`td[data-column="${column}"]`)?.textContent || '';
        
        return aValue.localeCompare(bValue);
    });
    
    rows.forEach(row => tbody.appendChild(row));
}

/**
 * Initialize pagination
 */
function initializePagination(pagination, table) {
    const links = pagination.querySelectorAll('a');
    
    links.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const page = this.getAttribute('data-page');
            loadTablePage(table, page);
        });
    });
}

/**
 * Load table page
 */
function loadTablePage(table, page) {
    // This would typically make an AJAX call to load the page
    console.log('Loading page:', page);
}

/**
 * Initialize notifications
 */
function initializeNotifications() {
    // Auto-hide notifications
    const notifications = document.querySelectorAll('.notification');
    
    notifications.forEach(notification => {
        setTimeout(() => {
            hideNotification(notification);
        }, 5000);
    });
}

/**
 * Show notification
 */
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    // Add animation
    setTimeout(() => notification.classList.add('show'), 10);
    
    // Auto-hide
    setTimeout(() => {
        hideNotification(notification);
    }, 5000);
}

/**
 * Hide notification
 */
function hideNotification(notification) {
    notification.classList.add('hide');
    
    setTimeout(() => {
        notification.remove();
    }, 300);
}

/**
 * Initialize search functionality
 */
function initializeSearch() {
    const searchInputs = document.querySelectorAll('.search-input');
    
    searchInputs.forEach(input => {
        input.addEventListener('input', debounce(function() {
            performSearch(this);
        }, 300));
    });
}

/**
 * Perform search
 */
function performSearch(input) {
    const searchTerm = input.value.trim();
    const searchUrl = input.getAttribute('data-search-url');
    
    if (searchUrl && searchTerm.length >= 2) {
        fetch(`${searchUrl}?q=${encodeURIComponent(searchTerm)}`, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            displaySearchResults(data, input);
        })
        .catch(error => {
            console.error('Search error:', error);
        });
    }
}

/**
 * Display search results
 */
function displaySearchResults(results, input) {
    const resultsContainer = document.querySelector(input.getAttribute('data-results-container'));
    
    if (resultsContainer) {
        resultsContainer.innerHTML = results.html || '';
    }
}

/**
 * Initialize modal dialogs
 */
function initializeModals() {
    const modalTriggers = document.querySelectorAll('[data-modal]');
    
    modalTriggers.forEach(trigger => {
        trigger.addEventListener('click', function(e) {
            e.preventDefault();
            const modalId = this.getAttribute('data-modal');
            openModal(modalId);
        });
    });
    
    // Close modal handlers
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal') || e.target.classList.contains('modal-close')) {
            closeModal(e.target.closest('.modal'));
        }
    });
    
    // Escape key to close modal
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            const openModal = document.querySelector('.modal.show');
            if (openModal) {
                closeModal(openModal);
            }
        }
    });
}

/**
 * Open modal
 */
function openModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Focus first input
        const firstInput = modal.querySelector('input, textarea, select');
        if (firstInput) {
            firstInput.focus();
        }
    }
}

/**
 * Close modal
 */
function closeModal(modal) {
    if (modal) {
        modal.classList.remove('show');
        document.body.classList.remove('modal-open');
    }
}

/**
 * Initialize file uploads
 */
function initializeFileUploads() {
    const fileInputs = document.querySelectorAll('input[type="file"]');
    
    fileInputs.forEach(input => {
        input.addEventListener('change', function() {
            handleFileUpload(this);
        });
    });
}

/**
 * Handle file upload
 */
function handleFileUpload(input) {
    const file = input.files[0];
    const preview = document.querySelector(input.getAttribute('data-preview'));
    
    if (file && preview) {
        if (file.type.startsWith('image/')) {
            const reader = new FileReader();
            reader.onload = function(e) {
                preview.src = e.target.result;
                preview.style.display = 'block';
            };
            reader.readAsDataURL(file);
        }
    }
}

/**
 * Initialize date pickers
 */
function initializeDatePickers() {
    const dateInputs = document.querySelectorAll('input[type="date"], input[type="datetime-local"]');
    
    dateInputs.forEach(input => {
        // Set default value if empty
        if (!input.value && input.getAttribute('data-default')) {
            input.value = input.getAttribute('data-default');
        }
    });
}

/**
 * Initialize auto-refresh
 */
function initializeAutoRefresh() {
    const refreshElements = document.querySelectorAll('[data-refresh]');
    
    refreshElements.forEach(element => {
        const interval = parseInt(element.getAttribute('data-refresh')) * 1000;
        
        setInterval(() => {
            refreshContent(element);
        }, interval);
    });
}

/**
 * Refresh content
 */
function refreshContent(element) {
    const url = element.getAttribute('data-refresh-url');
    
    if (url) {
        fetch(url, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.text())
        .then(html => {
            element.innerHTML = html;
        })
        .catch(error => {
            console.error('Refresh error:', error);
        });
    }
}

/**
 * Debounce function
 */
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

/**
 * Format number
 */
function formatNumber(num) {
    return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

/**
 * Format bytes
 */
function formatBytes(bytes, decimals = 2) {
    if (bytes === 0) return '0 Bytes';
    
    const k = 1024;
    const dm = decimals < 0 ? 0 : decimals;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
    
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    
    return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

/**
 * Export functions for global use
 */
window.AdminJS = {
    showNotification,
    hideNotification,
    openModal,
    closeModal,
    formatNumber,
    formatBytes,
    showTooltip,
    hideTooltip
};
